package com.esv.freight.customer.module.carrier.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.module.carrier.CarrierConstants;
import com.esv.freight.customer.module.carrier.dao.CarrierAccountDao;
import com.esv.freight.customer.module.carrier.dto.CarrierInfoDto;
import com.esv.freight.customer.module.carrier.entity.CarrierAccountEntity;
import com.esv.freight.customer.module.carrier.entity.CarrierInfoEntity;
import com.esv.freight.customer.module.carrier.form.CarrierInfoForm;
import com.esv.freight.customer.module.carrier.form.CarrierQueryForm;
import com.esv.freight.customer.module.carrier.service.CarrierAccountService;
import com.esv.freight.customer.module.carrier.service.CarrierInfoService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;


@Service("carrierAccountService")
public class CarrierAccountServiceImpl extends ServiceImpl<CarrierAccountDao, CarrierAccountEntity> implements CarrierAccountService {

    private FeignBaseService feignBaseService;

    private ErrorMessageComponent errorMessageComponent;

    private PasswordComponent passwordComponent;

    private CarrierInfoService carrierInfoService;

    @Autowired
    public CarrierAccountServiceImpl(FeignBaseService feignBaseService, ErrorMessageComponent errorMessageComponent,
                                     PasswordComponent passwordComponent, CarrierInfoService carrierInfoService) {
        this.feignBaseService = feignBaseService;
        this.errorMessageComponent = errorMessageComponent;
        this.passwordComponent = passwordComponent;
        this.carrierInfoService = carrierInfoService;
    }

    @Override
    public boolean isAccountExit(CarrierQueryForm form) {
        List<CarrierInfoDto> dtoList = this.baseMapper.selectCarrierList(form);
        if (null == dtoList || 0 == dtoList.size()) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public Long insertCarrier(CarrierInfoForm form) {
        // 1:校验
        // 1.1:校验帐号是否已存在
        CarrierQueryForm carrierQueryForm = new CarrierQueryForm();
        carrierQueryForm.setAccount(form.getAccount());
        if (this.isAccountExit(carrierQueryForm)) {
            throw new EException(1001, errorMessageComponent.getCarrierAccountAdd1001());
        }

        // 1.2:校验道路运输经营许可证号是否已存在
        carrierQueryForm = new CarrierQueryForm();
        carrierQueryForm.setRoadLicenseNumber(form.getRoadLicenseNumber());
        if (this.isAccountExit(carrierQueryForm)) {
            throw new EException(1002, errorMessageComponent.getCarrierAccountAdd1002());
        }

        // 承运商为企业
        if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(form.getCarrierType())) {
            // 1.3:校验统一社会信用代码是否已存在
            carrierQueryForm = new CarrierQueryForm();
            carrierQueryForm.setUniCreditCode(form.getUniCreditCode());
            if (this.isAccountExit(carrierQueryForm)) {
                throw new EException(1003, errorMessageComponent.getCarrierAccountAdd1003());
            }

            // 1.4:校验名称是否已存在
            carrierQueryForm = new CarrierQueryForm();
            carrierQueryForm.setCarrierFullName(form.getCarrierFullName());
            if (this.isAccountExit(carrierQueryForm)) {
                throw new EException(1004, errorMessageComponent.getCarrierAccountAdd1004());
            }
        }

        // 2:获取客户编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "CYS");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 15);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成承运商编号失败：" + e.getMessage());
            throw new EException("生成承运商编号时发生错误");
        }
        String carrierNumber = batchIdResJson.getString("batchId");

        // 3:新增帐号
        CarrierAccountEntity carrierAccountEntity = new CarrierAccountEntity();
        carrierAccountEntity.setAccount(form.getAccount());
        String salt = passwordComponent.generateAccountPwdSalt();
        carrierAccountEntity.setSalt(salt);
        carrierAccountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), salt));
        this.baseMapper.insert(carrierAccountEntity);
        Long accountId = carrierAccountEntity.getId();

        // 4:新增帐号信息
        CarrierInfoEntity carrierInfoEntity = new CarrierInfoEntity();
        BeanUtils.copyProperties(form, carrierInfoEntity);
        carrierInfoEntity.setAccountId(accountId);
        carrierInfoEntity.setCarrierNumber(carrierNumber);
        this.carrierInfoService.getBaseMapper().insert(carrierInfoEntity);

        return accountId;
    }

    @Override
    public Integer updateCarrier(CarrierInfoForm form) {
        int count = 0;
        // 1:校验
        // 1.1:校验帐号ID是否有效
        CarrierAccountEntity entity = this.baseMapper.selectById(form.getId());
        if (null == entity) {
            throw new EException(1001, errorMessageComponent.getCarrierAccountEdit1001());
        }

        // 1.2:校验道路运输经营许可证号是否已存在
        CarrierQueryForm carrierQueryForm = new CarrierQueryForm();
        carrierQueryForm.setId(form.getId());
        carrierQueryForm.setRoadLicenseNumber(form.getRoadLicenseNumber());
        if (this.isAccountExit(carrierQueryForm)) {
            throw new EException(1002, errorMessageComponent.getCarrierAccountEdit1002());
        }

        // 承运商为企业
        if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(form.getCarrierType())) {
            // 1.3:校验统一社会信用代码是否已存在
            carrierQueryForm = new CarrierQueryForm();
            carrierQueryForm.setId(form.getId());
            carrierQueryForm.setUniCreditCode(form.getUniCreditCode());
            if (this.isAccountExit(carrierQueryForm)) {
                throw new EException(1003, errorMessageComponent.getCarrierAccountEdit1003());
            }

            // 1.4:校验名称是否已存在
            carrierQueryForm = new CarrierQueryForm();
            carrierQueryForm.setId(form.getId());
            carrierQueryForm.setCarrierFullName(form.getCarrierFullName());
            if (this.isAccountExit(carrierQueryForm)) {
                throw new EException(1004, errorMessageComponent.getCarrierAccountEdit1004());
            }
        }

        // 2:更新帐号密码
        if (StringUtils.isNotBlank(form.getPassword())) {
            CarrierAccountEntity accountEntity = new CarrierAccountEntity();
            accountEntity.setId(form.getId());
            accountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), entity.getSalt()));
            count += this.baseMapper.updateById(accountEntity);
        }

        // 3:更新帐号信息
        CarrierInfoEntity carrierInfoEntity = new CarrierInfoEntity();
        BeanUtils.copyProperties(form, carrierInfoEntity);
        count += this.carrierInfoService.getBaseMapper().update(carrierInfoEntity,
                new UpdateWrapper<CarrierInfoEntity>().lambda().
                        eq(CarrierInfoEntity::getAccountId, form.getId()));

        return count;
    }

    @Override
    public CarrierInfoDto getCarrierDetail(CarrierQueryForm form) {
        CarrierAccountEntity entity = this.baseMapper.selectById(form.getId());
        if (null == entity) {
            throw new EException(1001, errorMessageComponent.getCarrierAccountDetail1001());
        }

        return this.baseMapper.selectCarrierDetail(form);
    }

    @Override
    public List<CarrierInfoDto> getAllCarrierBrief() {
        return this.baseMapper.selectAllCarrierBrief();
    }
}