package com.esv.freight.customer.module.contract.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.deepoove.poi.XWPFTemplate;
import com.deepoove.poi.data.PictureRenderData;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.RedisComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.common.util.InputStreamUtils;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.feign.FeignFileService;
import com.esv.freight.customer.module.contract.ContractConstants;
import com.esv.freight.customer.module.contract.dao.ContractOnlineRecordDao;
import com.esv.freight.customer.module.contract.entity.ContractOnlineRecordEntity;
import com.esv.freight.customer.module.contract.entity.ContractOnlineTemplateEntity;
import com.esv.freight.customer.module.contract.form.ContractOnlineRecordSignGoodsOwnerForm;
import com.esv.freight.customer.module.contract.pojo.ContractOnlineGoodsOwnerPojo;
import com.esv.freight.customer.module.contract.service.ContractOnlineRecordService;
import com.esv.freight.customer.module.contract.service.ContractOnlineTemplateService;
import com.esv.freight.customer.module.contract.vo.ContractOnlineRecordSignGoodsOwnerVO;
import com.esv.freight.customer.module.contract.vo.ContractOnlineRecordVO;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dto.AccountInfoDto;
import com.esv.freight.customer.module.goodsowner.form.AccountForm;
import com.esv.freight.customer.module.goodsowner.service.GoodsOwnerAccountService;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.Date;


@Service("contractOnlineRecordService")
public class ContractOnlineRecordServiceImpl extends ServiceImpl<ContractOnlineRecordDao, ContractOnlineRecordEntity> implements ContractOnlineRecordService {

    @Value("${spring.application.name}")
    private String applicationName;

    private static final Long CONTRACT_CACHE_TIME = 86400L;

    /**
     * 货主签名图像宽度、长度
     **/
    private static final int GOODS_OWNER_SIGN_WIDTH = 150;
    private static final int GOODS_OWNER_SIGN_HEIGHT = 60;

    private ErrorMessageComponent errorMessageComponent;
    private FeignFileService feignFileService;
    private FeignBaseService feignBaseService;
    private RedisComponent redisComponent;
    private ContractOnlineTemplateService contractOnlineTemplateService;
    private GoodsOwnerAccountService goodsOwnerAccountService;

    @Autowired
    public ContractOnlineRecordServiceImpl(ErrorMessageComponent errorMessageComponent, FeignFileService feignFileService,
                                           FeignBaseService feignBaseService, RedisComponent redisComponent,
                                           ContractOnlineTemplateService contractOnlineTemplateService, GoodsOwnerAccountService goodsOwnerAccountService) {
        this.errorMessageComponent = errorMessageComponent;
        this.feignFileService = feignFileService;
        this.feignBaseService = feignBaseService;
        this.redisComponent = redisComponent;
        this.contractOnlineTemplateService = contractOnlineTemplateService;
        this.goodsOwnerAccountService = goodsOwnerAccountService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractOnlineRecordSignGoodsOwnerVO goodsOwnerSign(ContractOnlineRecordSignGoodsOwnerForm form) {
        // 1:获取货主信息
        AccountForm accountForm = new AccountForm();
        accountForm.setId(form.getGoodsOwnerId());
        AccountInfoDto accountInfo = goodsOwnerAccountService.getAccountInfo(accountForm);
        if (null == accountInfo) {
            throw new EException(1001, errorMessageComponent.getContractOnlineGoodsOwnerAdd1001());
        }

        // 2:获取电子合同模板数据
        String contractTemplateData = this.getContractOnlineTemplateEntity(ContractConstants.CONTRACT_TYPE_GOODS_OWNER);
        byte[] contractTemplateBytes = Base64.getDecoder().decode(contractTemplateData);

        // 3:填充电子合同模板
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        ContractOnlineGoodsOwnerPojo contractOnlineGoodsOwnerPojo;
        byte[] signBytes;
        XWPFTemplate template = null;
        try {
            template = XWPFTemplate.compile(InputStreamUtils.byte2InputStream(contractTemplateBytes));
            contractOnlineGoodsOwnerPojo = this.getGoodsOwnerSignDataMap(form, accountInfo);
            template.render(contractOnlineGoodsOwnerPojo);
            template.write(byteArrayOutputStream);
            byteArrayOutputStream.flush();
            signBytes = byteArrayOutputStream.toByteArray();
        } catch (IOException e) {
            log.error("填充电子合同模板失败：" + e.getMessage());
            throw new EException(ECode.SERVER_ERROR.code(), "生成电子合同失败[填充电子合同模板失败]");
        } finally {
            if (null != template) {
                try {
                    template.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
            IOUtils.closeQuietly(byteArrayOutputStream);
        }

        // 4:保存货主签订后的电子合同文件
        String contractNumber = contractOnlineGoodsOwnerPojo.getContactNumber();
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("fileType", "word");
        feignReqJson.put("fileName", "电子合同-货主签名-" + contractNumber + ".docx");
        feignReqJson.put("fileData", Base64.getEncoder().encodeToString(signBytes));
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.uploadSingleFile(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]上传货主签名合同文件数据失败：" + e.getMessage());
            throw new EException("生成电子合同失败[上传货主签名合同文件数据失败]");
        }

        // 5:保存货主签订后的电子合同记录
        String fileId = feignResJson.getString("id");
        String fileUrl = feignResJson.getString("url");
        ContractOnlineRecordEntity onlineRecordEntity = new ContractOnlineRecordEntity();
        onlineRecordEntity.setContractType(ContractConstants.CONTRACT_TYPE_GOODS_OWNER);
        onlineRecordEntity.setContractFileId(fileId);
        onlineRecordEntity.setContractFileUrl(fileUrl);
        onlineRecordEntity.setContractNumber(contractNumber);
        onlineRecordEntity.setBusinessNumber(form.getBusinessNumber());
        onlineRecordEntity.setCustomerId(form.getGoodsOwnerId());
        onlineRecordEntity.setCustomerSignTime(new Date());
        onlineRecordEntity.setSignComplete(ContractConstants.CONTRACT_SIGN_COMPLETE_NO);
        this.baseMapper.insert(onlineRecordEntity);

        // 6:返回
        ContractOnlineRecordSignGoodsOwnerVO vo = new ContractOnlineRecordSignGoodsOwnerVO();
        vo.setContractNumber(contractNumber);
        return vo;
    }

    @Override
    public ContractOnlineRecordVO getContractInfoByNumber(String contractNumber) {
        ContractOnlineRecordEntity recordEntity = this.baseMapper.selectOne(new LambdaQueryWrapper<ContractOnlineRecordEntity>()
                .eq(ContractOnlineRecordEntity::getContractNumber, contractNumber));
        if (null == recordEntity) {
            throw new EException(1001, errorMessageComponent.getContractOnlineGetByNumber1001());
        }
        ContractOnlineRecordVO vo = new ContractOnlineRecordVO();
        BeanUtils.copyProperties(recordEntity, vo);

        // 判断是货主还是司机签订的合同，并获取客户的名称
        String customerName = null;
        if (ContractConstants.CONTRACT_TYPE_GOODS_OWNER.equals(recordEntity.getContractType())) {
            AccountForm accountForm = new AccountForm();
            accountForm.setId(recordEntity.getCustomerId());
            AccountInfoDto accountInfo = goodsOwnerAccountService.getAccountInfo(accountForm);
            if (null != accountInfo) {
                if (GoodsOwnerConstants.OWNER_TYPE_COMPANY.equals(accountInfo.getOwnerType())) {
                    customerName = accountInfo.getOwnerFullName();
                } else {
                    customerName = accountInfo.getContactor();
                }
            }
        } else {

        }
        vo.setCustomerName(customerName);
        vo.setCustomerSignTime(recordEntity.getCustomerSignTime().getTime());
        vo.setPlatformSignTime(null == recordEntity.getPlatformSignTime() ? null : recordEntity.getPlatformSignTime().getTime());
        vo.setEffectiveTime(null == recordEntity.getEffectiveTime() ? null : recordEntity.getEffectiveTime().getTime());

        return vo;
    }

    /**
     * description 获取货主签订合同时的数据
     * param [form, accountInfo]
     * return java.util.Map<java.lang.String,java.lang.Object>
     * author Administrator
     * createTime 2020/05/21 19:32
     **/
    private ContractOnlineGoodsOwnerPojo getGoodsOwnerSignDataMap(ContractOnlineRecordSignGoodsOwnerForm form, AccountInfoDto accountInfo) {
        ContractOnlineGoodsOwnerPojo pojo = new ContractOnlineGoodsOwnerPojo();
        BeanUtils.copyProperties(form, pojo);

        // 设置合同编号
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("prefix", "HZHT");
        feignReqJson.put("formatter", "yyyyMMdd");
        feignReqJson.put("length", 18);
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(feignReqJson));
            pojo.setContactNumber(feignResJson.getString("batchId"));
        } catch (Exception e) {
            log.error("调用[基础服务]获取合同编号失败：" + e.getMessage());
            throw new EException("生成电子合同失败[获取合同编号失败]");
        }

        // 设置货主信息
        if (GoodsOwnerConstants.OWNER_TYPE_COMPANY.equals(accountInfo.getOwnerType())) {
            pojo.setGoodsOwnerName(accountInfo.getOwnerFullName());
            pojo.setGoodsOwnerId(accountInfo.getUniCreditCode());
        } else {
            pojo.setGoodsOwnerName(accountInfo.getContactor());
            pojo.setGoodsOwnerId(accountInfo.getIdCard());
        }
        pojo.setPlatformName("{{platformName}}");
        pojo.setPlatformId("{{platformId}}");
        pojo.setEffectiveTime("{{effectiveTime}}");
        pojo.setGoodsOwnerSignImg(new PictureRenderData(GOODS_OWNER_SIGN_WIDTH, GOODS_OWNER_SIGN_HEIGHT, ".png", Base64.getDecoder().decode(form.getSignData())));
        pojo.setPlatformFreightSealImg("{{@platformFreightSealImg}}");

        return pojo;
    }

    /**
     * 获取电子合同模板数据
     * contractType：合同类型：1-货主与平台合同、2-司机与平台合同
     **/
    private String getContractOnlineTemplateEntity(Integer contractType) {
        String contractTemplateData;
        String cacheKey = applicationName + "::online-contract::" + contractType;
        contractTemplateData = (String) redisComponent.get(cacheKey);
        if (StringUtils.isBlank(contractTemplateData)) {
            ContractOnlineTemplateEntity contractOnlineTemplateEntity = this.contractOnlineTemplateService.getUsedTemplate(contractType);
            String fileId = contractOnlineTemplateEntity.getTemplateFileId();

            JSONObject feignReqJson = new JSONObject();
            feignReqJson.put("id", fileId);
            JSONObject feignResJson;
            try {
                feignResJson = FeignUtils.getFeignResultData(feignFileService.getFileData(feignReqJson));
            } catch (Exception e) {
                log.error("调用[文件服务]获取合同模板数据失败：" + e.getMessage());
                throw new EException("生成电子合同失败[获取合同模板数据失败]");
            }
            contractTemplateData = feignResJson.getString("fileData");
            redisComponent.set(cacheKey, contractTemplateData, CONTRACT_CACHE_TIME);
        }

        return contractTemplateData;
    }

}