package com.esv.superhive.cc.module.user.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.superhive.cc.common.cache.CacheNames;
import com.esv.superhive.cc.common.component.AccountPasswordComponent;
import com.esv.superhive.cc.common.exception.RRException;
import com.esv.superhive.cc.module.department.entity.DepartmentEntity;
import com.esv.superhive.cc.module.department.service.DepartmentService;
import com.esv.superhive.cc.module.user.dao.UserDao;
import com.esv.superhive.cc.module.user.dto.UserDTO;
import com.esv.superhive.cc.module.user.entity.UserEntity;
import com.esv.superhive.cc.module.user.entity.UserRoleEntity;
import com.esv.superhive.cc.module.user.service.UserRoleService;
import com.esv.superhive.cc.module.user.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.util.Map;


@Service("userService")
@Slf4j
@CacheConfig(cacheNames = {CacheNames.USER_INFO})
public class UserServiceImpl extends ServiceImpl<UserDao, UserEntity> implements UserService {

    private AccountPasswordComponent accountPasswordComponent;

    private DepartmentService departmentService;

    private UserRoleService userRoleService;

    @Autowired
    public UserServiceImpl(AccountPasswordComponent accountPasswordComponent, DepartmentService departmentService,
                           UserRoleService userRoleService) {
        this.accountPasswordComponent = accountPasswordComponent;
        this.departmentService = departmentService;
        this.userRoleService = userRoleService;
    }

    @Transactional
    @Override
    public UserEntity insert(UserEntity userEntity) {
        // 判断帐号是否存在
        UserEntity queryEntity = getUserByAccount(userEntity.getAccount());
        if (queryEntity != null) {
            throw new RRException("帐号已存在", 1001);
        }

        // 查询组织机构信息
        if (userEntity.getDepartmentId() != 0) {
            DepartmentEntity departmentEntity = departmentService.getById(userEntity.getDepartmentId());
            if (departmentEntity == null) {
                throw new RRException("无效的部门数据", 1002);
            } else {
                userEntity.setTenantId(departmentEntity.getTenantId());
            }
        }
        // 初始化帐号密码
        initAccountPwd(userEntity);
        // 新增帐号
        userEntity.setStatus(1);
        userEntity.setDeleted(false);
        baseMapper.insert(userEntity);
        return userEntity;
    }

    @CacheEvict(key = "#updateEntity.id")
    @Override
    public UserEntity updateUserInfoById(UserEntity updateEntity) {
        // 判断帐号是否存在
        QueryWrapper<UserEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("account", updateEntity.getAccount());
        queryWrapper.ne("id", updateEntity.getId());
        int activeAccountCount = baseMapper.selectCount(queryWrapper);
        if (0 < activeAccountCount) {
            throw new RRException("帐号已存在", 1001);
        }
        // 判断帐号是否存在
        UserEntity queryEntity = getById(updateEntity.getId());
        if (queryEntity == null) {
            throw new RRException("帐号不存在", 1001);
        }

        // 判断是否更新密码
        if (!StringUtils.isEmpty(updateEntity.getPassword())) {
            initAccountPwd(updateEntity);
        } else {
            updateEntity.setPassword(null);
        }

        // 查询组织机构信息
        if (updateEntity.getDepartmentId() != 0) {
            DepartmentEntity departmentEntity = departmentService.getById(updateEntity.getDepartmentId());
            if (departmentEntity == null) {
                throw new RRException("无效的部门数据", 1002);
            } else {
                updateEntity.setTenantId(departmentEntity.getTenantId());
            }
        }
        // 更新
        baseMapper.updateById(updateEntity);

        return updateEntity;
    }

    @Override
    public void getUserList(IPage<UserDTO> page, Map<String, Object> queryObj) {
        baseMapper.selectUserList(page, queryObj);
    }

    @Override
    public UserEntity getUserByAccount(String account) {
        UserEntity queryEntity = new UserEntity();
        queryEntity.setAccount(account);
        QueryWrapper<UserEntity> queryWrapper = new QueryWrapper<>(queryEntity);
        return baseMapper.selectOne(queryWrapper);
    }

    @Override
    public UserDTO getUserDetailInfo(Map<String, Object> queryObj) {
        return baseMapper.selectUserDetailInfo(queryObj);
    }

    /**
     * description 重写此方法增加使用缓存
     * param [id]
     * return UserEntity
     * author chenfm
     * createTime 2020/3/17 16:43
     **/
    @Cacheable
    @Override
    public UserEntity getById(Serializable id) {
        return super.getById(id);
    }

    /**
     * 初始化帐号密码
     * @param userEntity
     */
    private void initAccountPwd(UserEntity userEntity) {
        String salt = accountPasswordComponent.getSimpleSalt();
        userEntity.setSalt(salt);
        String reqPwd = userEntity.getPassword();
        userEntity.setPassword(accountPasswordComponent.generatePasswordDB(reqPwd, salt));
    }

    @CacheEvict
    @Override
    public boolean delete(long userId) {
        // 删除帐号角色
        UserRoleEntity userRoleEntity = new UserRoleEntity();
        userRoleEntity.setUserId(userId);
        userRoleService.remove(new QueryWrapper<>(userRoleEntity));
        // 删除账号
        return removeById(userId);
    }

}