package com.esv.superhive.cc.module.tenant.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.superhive.cc.common.constants.Constant;
import com.esv.superhive.cc.module.department.entity.DepartmentEntity;
import com.esv.superhive.cc.module.department.service.DepartmentService;
import com.esv.superhive.cc.module.department.vo.DepartmentInfoVO;
import com.esv.superhive.cc.module.department.vo.DepartmentVO;
import com.esv.superhive.cc.module.role.entity.RoleCategoryEntity;
import com.esv.superhive.cc.module.role.entity.RoleEntity;
import com.esv.superhive.cc.module.role.service.RoleCategoryService;
import com.esv.superhive.cc.module.role.service.RoleService;
import com.esv.superhive.cc.module.tenant.dao.TenantDao;
import com.esv.superhive.cc.module.tenant.dto.TenantAdminRoleDTO;
import com.esv.superhive.cc.module.tenant.entity.TenantEntity;
import com.esv.superhive.cc.module.tenant.form.TenantForm;
import com.esv.superhive.cc.module.tenant.service.TenantService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

@Service("tenantService")
public class TenantServiceImpl extends ServiceImpl<TenantDao, TenantEntity> implements TenantService {

    private DepartmentService departmentService;

    private RoleCategoryService roleCategoryService;

    private RoleService roleService;

    @Autowired
    public TenantServiceImpl(DepartmentService departmentService, RoleCategoryService roleCategoryService,
                             RoleService roleService) {
        this.departmentService = departmentService;
        this.roleCategoryService = roleCategoryService;
        this.roleService = roleService;
    }

    @Override
    public List<DepartmentVO> TenantDepartmentVOTreeList(Integer dataPerm, Long tenantId, String departmentName) {
        List<TenantEntity> tenantEntityList = listById(dataPerm, tenantId);
        List<DepartmentVO> departmentVOList = new ArrayList<>(tenantEntityList.size());
        tenantEntityList.forEach(tenantEntity -> {
            DepartmentVO departmentVO = new DepartmentVO(tenantEntity);
            List<DepartmentVO> children =
                    departmentService.listDepartmentTree(tenantEntity.getId(), 0, departmentName);
            children.forEach(child -> {
                child.setParentId(departmentVO.getId());
                departmentVO.setUserCount(departmentVO.getUserCount() + child.getUserCount());
            });
            departmentVO.setChildren(children);
            departmentVOList.add(departmentVO);
        });
        return departmentVOList;
    }

    @Override
    public List<TenantAdminRoleDTO> tenantAdminRoleList(Integer roleType) {
        return baseMapper.tenantAdminRoleList(roleType);
    }

    public List<TenantEntity> listById(Integer dataPerm, Long tenantId) {
        QueryWrapper<TenantEntity> wrapper = new QueryWrapper<>();
        wrapper.orderByAsc("order_num");
        if (dataPerm == null || dataPerm != 0) {
            List<Long> ids = new ArrayList<>();
            ids.add(tenantId);
            wrapper.in("id", ids);
        }
        return list(wrapper);
    }

    @Override
    public DepartmentInfoVO tenantInfo(Long tenantId) {
        TenantEntity tenantEntity = getById(tenantId);
        DepartmentInfoVO departmentInfoVO = new DepartmentInfoVO();
        departmentInfoVO.setId(String.valueOf(tenantEntity.getId()));
        departmentInfoVO.setDepartmentName(tenantEntity.getTenantName());
        departmentInfoVO.setParentId("0");
        departmentInfoVO.setTenantId(tenantEntity.getId());
        departmentInfoVO.setCategory("0");
        departmentInfoVO.setPrincipal(tenantEntity.getContact());
        departmentInfoVO.setPhone(tenantEntity.getContactPhone());
        return departmentInfoVO;
    }

    @Override
    public TenantEntity findByCode(String tenantCode) {
        TenantEntity entity = new TenantEntity();
        entity.setTenantCode(tenantCode);
        return baseMapper.selectOne(new QueryWrapper<>(entity));
    }

    @Override
    public boolean canDelete(Long tenantId) {
        DepartmentEntity departmentEntity = new DepartmentEntity();
        departmentEntity.setTenantId(tenantId);
        int departmentCount = departmentService.count(new QueryWrapper<>(departmentEntity));
        if (departmentCount != 0) {
            return false;
        }

        RoleCategoryEntity roleCategoryEntity = new RoleCategoryEntity();
        roleCategoryEntity.setTenantId(tenantId);
        int roleCategoryCount = roleCategoryService.count(new QueryWrapper<>(roleCategoryEntity));
        if (roleCategoryCount != 0) {
            return false;
        }

        RoleEntity roleEntity = new RoleEntity();
        roleEntity.setTenantId(tenantId);
        int roleCount = this.roleService.getBaseMapper().selectCount(new QueryWrapper<RoleEntity>().lambda()
                .eq(RoleEntity::getTenantId, tenantId)
                .ne(RoleEntity::getType, Constant.RoleType.TENANT_ADMIN.getValue()));

        if (roleCount != 0) {
            return false;
        }
        return true;
    }

    @Override
    public Boolean isTenantAdminRole(Long roleId) {
        TenantEntity entity = new TenantEntity();
        entity.setAdminRoleId(roleId);
        int count = baseMapper.selectCount(new QueryWrapper<>(entity));
        return count > 0;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addTenant(TenantForm tenantForm) {
        // 1:新增租户
        TenantEntity tenantEntity = new TenantEntity();
        BeanUtils.copyProperties(tenantForm, tenantEntity);
        this.baseMapper.insert(tenantEntity);

        // 2:新增角色，关联租户ID
        Long roleId = roleService.createTenantAdminRole(tenantEntity);

        // 3:更新租户：设置租户管理员角色ID
        tenantEntity.setAdminRoleId(roleId);
        // 修改排序值与id相同
        tenantEntity.setOrderNum(tenantEntity.getId().intValue());
        this.baseMapper.updateById(tenantEntity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteTenant(Long id) {
        // 1:删除租户
        TenantEntity tenantEntity = this.getBaseMapper().selectById(id);
        this.baseMapper.deleteById(id);

        // 2:删除租户管理员角色
        this.roleService.getBaseMapper().deleteById(tenantEntity.getAdminRoleId());
    }
}