package com.esv.superhive.cc.module.department.controller;

import com.esv.superhive.cc.common.form.Add;
import com.esv.superhive.cc.common.form.Update;
import com.esv.superhive.cc.module.department.entity.DepartmentEntity;
import com.esv.superhive.cc.module.department.form.DepartmentDeleteForm;
import com.esv.superhive.cc.module.department.form.DepartmentForm;
import com.esv.superhive.cc.module.department.service.DepartmentService;
import com.esv.superhive.cc.module.department.vo.DepartmentChooseVO;
import com.esv.superhive.cc.module.department.vo.DepartmentInfoVO;
import com.esv.superhive.cc.module.department.vo.DepartmentVO;
import com.esv.superhive.cc.module.tenant.service.TenantService;
import com.esv.common.exception.EException;
import com.esv.common.response.R;
import com.esv.gateway.common.GatewayHeaders;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 部门表
 *
 * @author chenfm
 * @email chenfengman@esvtek.com
 * @date 2020-01-09 11:32:08
 */
@Slf4j
@RestController
@RequestMapping("department")
public class DepartmentController {

    private DepartmentService departmentService;

    private TenantService tenantService;

    @Autowired
    public DepartmentController(DepartmentService departmentService, TenantService tenantService) {
        this.departmentService = departmentService;
        this.tenantService = tenantService;
    }

    /**
     * description 查询租户部门树列表
     * param [tenantId, departmentName]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/17 13:37
     **/
    @GetMapping("listTree")
    public R listTree(
            @RequestHeader(GatewayHeaders.DATA_PERM) Integer dataPerm,
            @RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
            @RequestHeader(GatewayHeaders.DEPARTMENT_ID) Long departmentId,
            @RequestParam(value = "departmentName", required = false) String departmentName
    ) {
        // 以租户信息为要结点查询部门信息
        List<DepartmentVO> departmentVOList;
        if ((dataPerm == 2) && (departmentId != 0)) {
            // departmentId = 0表示该部门是租户级
            departmentVOList = new ArrayList<>(1);
            DepartmentEntity departmentEntity = departmentService.getById(departmentId);
            DepartmentVO departmentVO = new DepartmentVO(departmentEntity);
            departmentVO.setChildren(departmentService.listDepartmentTree(tenantId, departmentId, departmentName));
            departmentVOList.add(departmentVO);
        } else {
            departmentVOList = tenantService.TenantDepartmentVOTreeList(dataPerm, tenantId, departmentName);
        }
        return R.ok(departmentVOList);
    }

    /**
     * description 查询部门详细信息
     * param [departmentId]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/17 13:39
     **/
    @GetMapping("info")
    public R info(@RequestParam("departmentId") String departmentId) {
        DepartmentInfoVO departmentInfoVO;
        if (departmentId.startsWith("t_")) {
            // 如果查询部门id以t_开头说明为租户
            Long tenantId = Long.parseLong(departmentId.split("_")[1]);
            departmentInfoVO = tenantService.tenantInfo(tenantId);
        } else {
            // 查询部门信息
            departmentInfoVO = departmentService.departmentInfo(Long.parseLong(departmentId));
        }
        if (departmentInfoVO == null) {
            return R.error("没有数据");
        }
        return R.ok(departmentInfoVO);
    }

    /**
     * description 新增部门信息
     * param [userId, departmentForm]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/17 13:40
     **/
    @PostMapping("add")
    public R add(@RequestBody @Validated(Add.class) DepartmentForm departmentForm) {
        DepartmentEntity departmentEntity = new DepartmentEntity();
        long parentId;
        long tenantId;
        if (departmentForm.getParentId().startsWith("t_")) {
            // 加入租户的根部门
            parentId = 0;
            tenantId = Long.parseLong(departmentForm.getParentId().split("_")[1]);
        } else {
            // 加入子部门
            parentId = Long.parseLong(departmentForm.getParentId());
            DepartmentEntity entity = departmentService.getById(parentId);
            if (parentId != 0 && entity == null) {
                return R.error("该上级组织不存在");
            }
            tenantId = entity.getTenantId();
        }
        BeanUtils.copyProperties(departmentForm, departmentEntity);
        departmentEntity.setParentId(parentId);
        departmentEntity.setTenantId(tenantId);

        if (departmentService.save(departmentEntity)) {
            return R.ok();
        } else {
            return R.error();
        }
    }

    /**
     * description 修改时可选择部门列表
     * param [departmentId]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/17 13:40
     **/
    @GetMapping("chooseList")
    public R chooseList(
            @RequestHeader(GatewayHeaders.DATA_PERM) Integer dataPerm,
            @RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
            @RequestHeader(GatewayHeaders.DEPARTMENT_ID) Long departmentId,
            @RequestParam("departmentId") String chooseDepartmentId) {
        List<DepartmentVO> departmentVOList;
        if ((dataPerm == 2) && (departmentId != 0)) {
            // departmentId = 0表示该部门是租户级
            departmentVOList = new ArrayList<>(1);
            DepartmentEntity departmentEntity = departmentService.getById(departmentId);
            DepartmentVO departmentVO = new DepartmentVO(departmentEntity);
            departmentVO.setChildren(departmentService.listDepartmentTree(tenantId, departmentId, null));
            departmentVOList.add(departmentVO);
        } else {
            Integer queryDataPerm = dataPerm;
            Long queryTenantId = tenantId;
            // 查询要修改部门的租户
            if (StringUtils.isNotBlank(chooseDepartmentId)) {
                queryDataPerm = null;
                DepartmentEntity entity = departmentService.getById(chooseDepartmentId);
                queryTenantId = entity.getTenantId();
            }
            departmentVOList = tenantService.TenantDepartmentVOTreeList(queryDataPerm, queryTenantId, null);
        }
        List<DepartmentChooseVO> list = new ArrayList<>(departmentVOList.size());
        for (DepartmentVO departmentVO : departmentVOList) {
            list.add(departmentService.makeChooseVO(chooseDepartmentId, departmentVO));
        }
        return R.ok(list);
    }

    /**
     * description 修改部门信息
     * param [userId, departmentUpdateForm]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/17 13:40
     **/
    @PostMapping("update")
    public R update(@RequestBody @Validated(Update.class) DepartmentForm departmentForm) {
        long id = departmentForm.getId();
        DepartmentEntity departmentEntity = departmentService.getById(id);
        long tenantId = departmentEntity.getTenantId();
        long parentId;
        if (departmentForm.getParentId().startsWith("t_")) {
            // 修改为租户根部门
            parentId = 0;
        } else {
            // 修改为子部门
            parentId = Long.parseLong(departmentForm.getParentId());
            if (departmentService.getById(parentId) == null) {
                return R.error("该上级组织不存在");
            }
            // 检查上级部门是否为当前及下级部门
            List<DepartmentVO> departmentVOList =
                    departmentService.listDepartmentTree(tenantId, id, null);
            if ((id == parentId) || treeHasId(departmentVOList, parentId)) {
                return R.error("该上级组织不可更改为当前及下级组织");
            }
        }
        BeanUtils.copyProperties(departmentForm, departmentEntity);
        departmentEntity.setParentId(parentId);

        if (departmentService.updateById(departmentEntity)) {
            return R.ok();
        } else {
            return R.error();
        }
    }

    /**
     * description 树列表是否包括指定id
     * param [departmentVOList, parentId]
     * return boolean
     * author chenfm
     * createTime 2020/3/17 13:41
     **/
    private boolean treeHasId(List<DepartmentVO> departmentVOList, long parentId) {
        for (DepartmentVO departmentVO : departmentVOList) {
            if (Long.parseLong(departmentVO.getId()) == parentId) {
                return true;
            }
            if (departmentVO.getChildren() != null) {
                if(treeHasId(departmentVO.getChildren(), parentId)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * description 删除部门
     * param [userId, departmentDeleteForm]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/17 13:41
     **/
    @PostMapping("delete")
    public R delete(@RequestBody @Validated DepartmentDeleteForm departmentDeleteForm) {
        if (departmentDeleteForm.getDepartmentId().startsWith("t_")) {
            return R.error("根部门不可删除");
        }
        try {
            departmentService.deleteById(Long.parseLong(departmentDeleteForm.getDepartmentId()));
        } catch (EException e) {
            return R.error(e.getMsg());
        }
        return R.ok();
    }

}
