package com.esv.superhive.cc.module.account.service.impl;

import com.esv.superhive.cc.module.account.entity.UserLoginInfoEntity;
import com.esv.superhive.cc.module.account.service.AccountService;
import com.esv.superhive.cc.module.account.service.UserLoginInfoService;
import com.esv.superhive.cc.common.component.AccountPasswordComponent;
import com.esv.superhive.cc.common.component.JwtAuthComponent;
import com.esv.superhive.cc.common.exception.RRException;
import com.esv.superhive.cc.module.account.bo.TokenBO;
import com.esv.superhive.cc.module.user.entity.UserEntity;
import com.esv.superhive.cc.module.user.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cglib.beans.BeanMap;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.Map;

/**
 * @description： 帐户中心服务实现类
 * @author：hcbmailbox@163.com
 * @date：2020/1/15
 */
@Service("accountService")
@Slf4j
@RefreshScope
public class AccountServiceImpl implements AccountService {
    /**
     * Token存活时间，单位：分钟
     */
    @Value("${token.live.time.web:0}")
    private Long tokenLiveTimeWeb;

    private AccountPasswordComponent accountPasswordComponent;

    private JwtAuthComponent jwtAuthComponent;

    private UserService userService;

    private UserLoginInfoService userLoginInfoService;

    @Autowired
    public AccountServiceImpl(AccountPasswordComponent accountPasswordComponent, JwtAuthComponent jwtAuthComponent,
                              UserService userService, UserLoginInfoService userLoginInfoService) {
        this.accountPasswordComponent = accountPasswordComponent;
        this.jwtAuthComponent = jwtAuthComponent;
        this.userService = userService;
        this.userLoginInfoService = userLoginInfoService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String loginByAccountPwd(String account, String pwd, Map<String, Object> extMap) {
        // 帐号密码校验
        UserEntity userEntity = userService.getUserByAccount(account);
        if (null == userEntity) {
            throw new RRException("帐号不存在", 1001);
        }
        String salt = userEntity.getSalt();
        String reqPwd = accountPasswordComponent.generatePasswordDB(pwd, salt);
        if (!reqPwd.equals(userEntity.getPassword())) {
            throw new RRException("密码错误", 1002);
        }

        // 创建Token
        TokenBO tokenBO = new TokenBO();
        tokenBO.setUserId(userEntity.getId());
        tokenBO.setUserAccount(userEntity.getAccount());
        tokenBO.setType(userEntity.getType());
        tokenBO.setTenantId(userEntity.getTenantId());
        tokenBO.setDepartmentId(userEntity.getDepartmentId());
        tokenBO.setCreateTime(System.currentTimeMillis());
        if (0L == tokenLiveTimeWeb) {
            tokenBO.setExpireTime(0L);
        } else {
            // Token存活时间
            Long expireTime = System.currentTimeMillis() + (tokenLiveTimeWeb * 60 * 60 * 1000);
            tokenBO.setExpireTime(expireTime);
        }
        tokenBO.setIp(String.valueOf(extMap.get("ip")));
        tokenBO.setSource("web");

        // 将bean转换成map
        BeanMap beanMap = BeanMap.create(tokenBO);
        // 创建Token
        String token = jwtAuthComponent.generateToken(beanMap);
        log.info("帐号[{}]创建访问Token成功", account);

        // 保存登录信息
        UserLoginInfoEntity loginInfoEntity = new UserLoginInfoEntity();
        loginInfoEntity.setUserId(userEntity.getId());
        loginInfoEntity.setIp(String.valueOf(extMap.get("ip")));
        loginInfoEntity.setToken(token);
        loginInfoEntity.setStatus("0");
        loginInfoEntity.setLoginTime(new Date());
        loginInfoEntity.setLogoutTime(null);
        userLoginInfoService.saveLoginInfo(loginInfoEntity);

        return token;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void logout(TokenBO tokenBO) {
        // 保存登出信息
        UserLoginInfoEntity loginInfoEntity = new UserLoginInfoEntity();
        loginInfoEntity.setUserId(tokenBO.getUserId());
        loginInfoEntity.setStatus("1");
        loginInfoEntity.setLogoutTime(new Date());
        userLoginInfoService.saveLoginInfo(loginInfoEntity);

        String account = tokenBO.getUserAccount();
        log.info("帐号[{}]通过token登出成功", account);
    }

}
