package com.esv.superhive.cc.module.menu.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.superhive.cc.common.cache.CacheNames;
import com.esv.superhive.cc.module.menu.dao.MenuDao;
import com.esv.superhive.cc.module.menu.dto.MenuDTO;
import com.esv.superhive.cc.module.menu.dto.SystemMenuDTO;
import com.esv.superhive.cc.module.menu.entity.MenuEntity;
import com.esv.superhive.cc.module.menu.form.MenuForm;
import com.esv.superhive.cc.module.menu.service.MenuService;
import com.esv.superhive.cc.module.role.service.RoleMenuService;
import com.esv.superhive.cc.module.user.service.UserRoleService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.io.Serializable;
import java.util.*;

@Service("menuService")
public class MenuServiceImpl extends ServiceImpl<MenuDao, MenuEntity> implements MenuService {

    private UserRoleService userRoleService;

    private RoleMenuService roleMenuService;

    @Autowired
    public MenuServiceImpl(UserRoleService userRoleService, RoleMenuService roleMenuService) {
        this.userRoleService = userRoleService;
        this.roleMenuService = roleMenuService;
    }

    @Cacheable(cacheNames = CacheNames.MENU_INFO)
    @Override
    public MenuEntity getById(Serializable id) {
        return super.getById(id);
    }

    @Override
    public List<SystemMenuDTO> listSystemMenuTree(Long userId) {
        boolean hasSuperAdminRole = userRoleService.hasSuperAdminRole(userId);
        Long queryUserId = null;
        if (!hasSuperAdminRole) {
            // 非超级管理员需要根据用户权限查询菜单
            queryUserId = userId;
        }
        // 系统名称作为菜单树根结点
        List<SystemMenuDTO> sysRootMenuDTOList = baseMapper.listSystemMenuTree(queryUserId);
        for (SystemMenuDTO systemMenuDTO : sysRootMenuDTOList) {
            for (MenuDTO menuDTO : systemMenuDTO.getChildren()) {
                menuDTO.setParentName(systemMenuDTO.getName());
                // 查询系统信息下的菜单树
                List<MenuDTO> menuTreeList = getMenuTreeList(menuDTO.getId(), queryUserId);
                menuDTO.setChildren(menuTreeList);
            }
        }
        return sysRootMenuDTOList;
    }

    @Override
    public boolean saveOrUpdate(MenuForm menuForm) {
        MenuEntity menuEntity = new MenuEntity();
        BeanUtils.copyProperties(menuForm, menuEntity);
        long parentId = parseIdByRowKey(menuForm.getParentRowKey());
        if (menuForm.getType() == 0) {
            // 菜单类别为目录, 上级菜单传入为系统id
            menuEntity.setParentId(0L);
            menuEntity.setSystemId(parentId);
        } else {
            // 菜单类别为菜单或按钮, 系统id应为上级菜单的系统id
            menuEntity.setParentId(parentId);
            MenuEntity parent = getById(parentId);
            menuEntity.setSystemId(parent.getSystemId());
        }
        return saveOrUpdate(menuEntity);
    }

    @Override
    public Long parseIdByRowKey(String rowKey) {
        return Long.valueOf(rowKey.split("_")[0]);
    }

    private List<MenuDTO> getMenuTreeList(long parentId, Long userId) {
        List<MenuDTO> menuTreeList = baseMapper.queryByParentId(parentId, userId);
        if (menuTreeList != null) {
            for (MenuDTO menuDTO : menuTreeList) {
                // 如果是目录或菜单继续查询子菜单或按钮
                if (menuDTO.getType() == 0 || menuDTO.getType() == 1) {
                    menuDTO.setChildren(getMenuTreeList(menuDTO.getId(), userId));
                    if (menuDTO.getType() == 1) {
                        // 将管理员菜单数据权限设为全部
                        if (userId == null) {
                            menuDTO.setUserDataPerm(0);
                        } else {
                            Integer dataPerm = roleMenuService.findUserMenuCodeDataPerm(userId, menuDTO.getMenuCode());
                            menuDTO.setUserDataPerm(dataPerm == null ? 2 : dataPerm);
                        }
                    }
                }
            }
        }
        return menuTreeList;
    }

    @CacheEvict(cacheNames = CacheNames.MENU_INFO)
    @Override
    public void removeMenuTree(long menuId) {
        List<MenuDTO> menuTreeList = baseMapper.queryByParentId(menuId, null);
        for (MenuDTO menuDTO : menuTreeList) {
            // 如果是目录或菜单继续删除子菜单或按钮
            if (menuDTO.getType() == 0 || menuDTO.getType() == 1) {
                removeMenuTree(menuDTO.getId());
            }
        }
        removeMenu(menuId);
    }

    private void removeMenu(long menuId) {
        // 删除权限中关联的菜单
        roleMenuService.deleteByMenuId(menuId);
        baseMapper.deleteById(menuId);
    }

    @Override
    public List<MenuDTO> listSystemMenu(Long systemId, Long userId, Long tenantId) {
        Map<String, Object> param = new HashMap<>();
        param.put("systemId", systemId);
        param.put("parentId", 0);
        param.put("type", 0);
        List<MenuDTO> systemTypeMenuList = baseMapper.systemTypeMenu(param);

        if (userId != null) {
            param.put("userId", userId);
        }
        if (tenantId != null) {
            param.put("tenantId", tenantId);
        }
        param.put("type", 1);
        param.remove("systemId");
        List<MenuDTO> list = new ArrayList<>();
        List<MenuDTO> children;
        for (MenuDTO menuDTO : systemTypeMenuList) {
            param.put("parentId", menuDTO.getId());
            children = baseMapper.querySystemMenu(param);
            if (children != null && !children.isEmpty()) {
                menuDTO.setChildren(children);
                list.add(menuDTO);
            }
        }
        return list;
    }

    @Override
    public List<String> listUserPerms(Long systemId, Long userId, Long tenantId) {
        Map<String, Object> param = new HashMap<>(3);
        param.put("systemId", systemId);
        if (userId != null) {
            param.put("userId", userId);
        }
        if (tenantId != null) {
            param.put("tenantId", tenantId);
        }
        List<String> permsList = baseMapper.queryPermsByUser(param);
        return permsList;
    }

    @Override
    public List<String> listMenuCode(Long systemId, Long userId, Long tenantId) {
        Map<String, Object> param = new HashMap<>();
        param.put("systemId", systemId);
        param.put("type", 2);
        if (userId != null) {
            param.put("userId", userId);
        }
        if (tenantId != null) {
            param.put("tenantId", tenantId);
        }
        List<String> menuCodeList = baseMapper.queryMenuCodeByUser(param);
        return menuCodeList;
    }

    @Override
    @Cacheable(value = CacheNames.SYSTEM_USER_PERM)
    public Set<String> userPermSet(Long systemId, Long userId, Long tenantId) {
        List<String> permsList = this.listUserPerms(systemId, userId, tenantId);
        Set<String> permSet = new HashSet<>();
        for (String perms : permsList) {
            if (StringUtils.isNotBlank(perms)) {
                permSet.addAll(Arrays.asList(perms.split(";")));
            }
        }
        return permSet;
    }

    @Override
    public void deleteBySystemId(Long systemId) {
        MenuEntity entity = new MenuEntity();
        entity.setSystemId(systemId);
        List<MenuEntity> list = list(new QueryWrapper<>(entity));
        for (MenuEntity menuEntity : list) {
            removeMenu(menuEntity.getId());
        }
    }
}