package com.esv.superhive.cc.module.account.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.superhive.cc.common.component.LoginMBean;
import com.esv.superhive.cc.common.component.JwtAuthComponent;
import com.esv.superhive.cc.common.constants.ApiResponseCode;
import com.esv.superhive.cc.feign.client.CaptchaFeignClient;
import com.esv.superhive.cc.feign.req.CaptchaVerifyReq;
import com.esv.superhive.cc.feign.res.CaptchaVerifyRes;
import com.esv.superhive.cc.module.account.bo.TokenBO;
import com.esv.superhive.cc.module.account.dto.UserLoginInfoDTO;
import com.esv.superhive.cc.module.account.dto.UserSystemDTO;
import com.esv.superhive.cc.module.account.form.AccountForm;
import com.esv.superhive.cc.module.account.service.AccountService;
import com.esv.superhive.cc.module.account.service.UserLoginInfoService;
import com.esv.superhive.cc.module.account.vo.LoginVO;
import com.esv.superhive.cc.module.system.service.SystemService;
import com.esv.superhive.cc.util.ReqUtils;
import com.esv.common.response.ECode;
import com.esv.common.response.EResponse;
import com.esv.common.response.R;
import com.esv.gateway.common.GatewayHeaders;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @description： 帐号服务Controller
 * @author：hcbmailbox@163.com
 * @date：2020/1/15
 */
@RestController
@RequestMapping("account")
@Slf4j
public class AccountController {

    private CaptchaFeignClient captchaFeignClient;

    private JwtAuthComponent jwtAuthComponent;

    private AccountService accountService;

    private UserLoginInfoService userLoginInfoService;

    private HttpServletRequest request;

    private LoginMBean loginMBean;

    private SystemService systemService;

    @Autowired
    public AccountController(CaptchaFeignClient captchaFeignClient, JwtAuthComponent jwtAuthComponent,
                             AccountService accountService,
                             UserLoginInfoService userLoginInfoService, HttpServletRequest request,
                             LoginMBean loginMBean, SystemService systemService) {
        this.captchaFeignClient = captchaFeignClient;
        this.jwtAuthComponent = jwtAuthComponent;
        this.accountService = accountService;
        this.userLoginInfoService = userLoginInfoService;
        this.request = request;
        this.loginMBean = loginMBean;
        this.systemService = systemService;
    }

    /**
     * 帐号密码登录
     * @return
     * @throws Exception
     */
    @PostMapping("loginByAccountPwd")
    public EResponse<LoginVO> loginByAccountPwd(HttpServletRequest request, @RequestBody @Validated AccountForm form) {
        CaptchaVerifyReq captchaVerifyReq = new CaptchaVerifyReq(form.getId(), form.getCaptcha());
        JSONObject json = captchaFeignClient.captchaVerify(captchaVerifyReq);
        CaptchaVerifyRes captchaVerifyRes = json.getObject("data", CaptchaVerifyRes.class);
        if (!captchaVerifyRes.getResult()) {
            return EResponse.error(ApiResponseCode.CAPTCHA_ERROR);
        }

        Map<String, Object> extMap = new HashMap<>(1);
        extMap.put("ip", StringUtils.trimToEmpty(ReqUtils.getHttpClientIp(request)));
        String token = accountService.loginByAccountPwd(form.getAccount(), form.getPassword(), extMap);

        LoginVO loginVO = new LoginVO();
        loginVO.setToken(token);
        try {
            loginMBean.setCount(loginMBean.getCount() + 1);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return EResponse.ok(loginVO);
    }

    /**
     * 帐号登出
     * @return
     * @throws Exception
     */
    @PostMapping("logout")
    public EResponse logout() {
        String token = jwtAuthComponent.getRequestToken(this.request);
        TokenBO tokenBO = jwtAuthComponent.parseToken(token);
        if (null == tokenBO) {
            return EResponse.error(ECode.TOKEN_INVALID);
        } else {
            accountService.logout(tokenBO);
            return EResponse.ok();
        }
    }

    /**
     * Token有效性校验
     * @param request
     * @return
     * @throws Exception
     */
    @GetMapping("checkToken")
    public EResponse checkToken(HttpServletRequest request) {
        String token = jwtAuthComponent.getRequestToken(request);
        if (StringUtils.isEmpty(token)) {
            return EResponse.error(ECode.TOKEN_INVALID);
        }
        TokenBO tokenBO = jwtAuthComponent.parseToken(token);
        if (null == tokenBO) {
            return EResponse.error(ECode.TOKEN_INVALID);
        }
        if (0L != tokenBO.getExpireTime() && System.currentTimeMillis() > tokenBO.getExpireTime()) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }
        return EResponse.ok();
    }

    /**
     * 获取用户登录信息
     * @param userId
     * @return
     */
    @GetMapping("getUserLoginInfo")
    public R getUserLoginInfo(@RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
                              @RequestHeader(GatewayHeaders.USER_ID) Long userId) {
        UserLoginInfoDTO userLoginInfoDTO = userLoginInfoService.getUserLoginInfo(tenantId, userId);
        List<UserSystemDTO> userSystemList = systemService.userSystemList(tenantId, userId);
        userLoginInfoDTO.setUserSystemList(userSystemList);
        return R.ok(userLoginInfoDTO);
    }
}
