package com.esv.datacenter.cc.module.department.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.datacenter.cc.module.department.dao.DepartmentDao;
import com.esv.datacenter.cc.module.department.dto.DepartmentDTO;
import com.esv.datacenter.cc.module.department.entity.DepartmentEntity;
import com.esv.datacenter.cc.module.department.service.DepartmentService;
import com.esv.datacenter.cc.module.department.vo.DepartmentChooseVO;
import com.esv.datacenter.cc.module.department.vo.DepartmentInfoVO;
import com.esv.datacenter.cc.module.department.vo.DepartmentVO;
import com.esv.datacenter.cc.module.user.dao.UserDao;
import com.esv.datacenter.cc.module.user.entity.UserEntity;
import com.esv.common.exception.EException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * @author chenfm
 */
@Service("departmentService")
public class DepartmentServiceImpl extends ServiceImpl<DepartmentDao, DepartmentEntity> implements DepartmentService {

    private UserDao userDao;

    @Autowired
    public DepartmentServiceImpl(UserDao userDao) {
        this.userDao = userDao;
    }

    /**
     * description 递归查询部门树列表
     * param [tenantId, parentId, departmentName]
     * return java.util.List<DepartmentDTO>
     * author chenfm
     * createTime 2020/3/11 13:55
     **/
    @Override
    public List<DepartmentVO> listDepartmentTree(long tenantId, long parentId, String departmentName) {
        List<DepartmentDTO> list = baseMapper.selectByParentId(tenantId, parentId, departmentName);
        List<DepartmentVO> departmentVOList = new ArrayList<>(list.size());
        for (DepartmentDTO dto : list) {
            DepartmentVO departmentVO = new DepartmentVO();
            BeanUtils.copyProperties(dto, departmentVO);
            departmentVO.setId(String.valueOf(dto.getId()));
            departmentVO.setParentId(String.valueOf(dto.getParentId()));

            int userCount = dto.getUserCount();
            List<DepartmentVO> children = listDepartmentTree(tenantId, dto.getId(), departmentName);
            for (DepartmentVO child : children) {
                userCount += child.getUserCount();
            }
            departmentVO.setChildren(children);
            departmentVO.setUserCount(userCount);
            departmentVOList.add(departmentVO);
        }
        return departmentVOList;
    }

    @Override
    public DepartmentInfoVO departmentInfo(Long departmentId) {
        DepartmentEntity departmentEntity = getById(departmentId);
        if (departmentEntity == null) {
            return null;
        }
        DepartmentInfoVO departmentInfoVO = new DepartmentInfoVO();
        BeanUtils.copyProperties(departmentEntity, departmentInfoVO);
        departmentInfoVO.setId(String.valueOf(departmentEntity.getId()));
        if (departmentEntity.getParentId() == 0) {
            departmentInfoVO.setParentId("t_" + departmentEntity.getTenantId());
        } else {
            departmentInfoVO.setParentId(String.valueOf(departmentEntity.getParentId()));
        }
        List<String> parentNameList = getParentNameList(departmentEntity.getParentId());
        if (!parentNameList.isEmpty()) {
            departmentInfoVO.setParentName(parentNameList.get(parentNameList.size() - 1));
            departmentInfoVO.setParentNameList(parentNameList);
        }
        return departmentInfoVO;
    }

    @Override
    public List<String> getParentNameList(long departmentId) {
        List<String> list = new ArrayList<>();
        if (departmentId == 0) {
            return list;
        }
        DepartmentEntity entity = baseMapper.selectById(departmentId);
        list.addAll(getParentNameList(entity.getParentId()));
        list.add(entity.getDepartmentName());
        return list;
    }

    @Transactional
    @Override
    public void deleteById(long departmentId) {
        DepartmentEntity entity = new DepartmentEntity();
        entity.setParentId(departmentId);
        List<DepartmentEntity> list = list(new QueryWrapper<>(entity));
        for (DepartmentEntity departmentEntity : list) {
            deleteById(departmentEntity.getId());
        }
        // 判断要删除的部门中是否有用户
        UserEntity userEntity = new UserEntity();
        userEntity.setDepartmentId(departmentId);
        int userCount = userDao.selectCount(new QueryWrapper<>(userEntity));
        if (userCount != 0) {
            throw new EException("该部门或下级部门存在用户");
        }
        removeById(departmentId);
    }

    @Override
    public List<DepartmentChooseVO> chooseList(Long tenantId, Long parentId, String departmentId) {
        List<DepartmentVO> list = this.listDepartmentTree(tenantId, parentId, null);
        List<DepartmentChooseVO> chooseVOList = new ArrayList<>();
        for (DepartmentVO departmentVO : list) {
            DepartmentChooseVO departmentChooseVO = makeChooseVO(departmentId, departmentVO);
            if (departmentChooseVO != null) {
                chooseVOList.add(departmentChooseVO);
            }
        }
        return chooseVOList;
    }

    public DepartmentChooseVO makeChooseVO(String departmentId, DepartmentVO departmentVO) {
        if (StringUtils.isNotBlank(departmentId) && departmentVO.getId().equals(departmentId)) {
            return null;
        }
        DepartmentChooseVO departmentChooseVO = new DepartmentChooseVO();
        departmentChooseVO.setId(departmentVO.getId());
        departmentChooseVO.setDepartmentName(departmentVO.getDepartmentName());
        if (departmentVO.getChildren() != null) {
            List<DepartmentChooseVO> chooseVOList = new ArrayList<>();
            for (DepartmentVO child : departmentVO.getChildren()) {
                DepartmentChooseVO childVO = makeChooseVO(departmentId, child);
                if (childVO != null) {
                    chooseVOList.add(childVO);
                }
            }
            if (!chooseVOList.isEmpty()) {
                departmentChooseVO.setChildren(chooseVOList);
            }
        }
        return departmentChooseVO;
    }

    @Override
    public void findCurrentAndChildrenId(List<Long> departmentIdList, Long departmentId) {
        departmentIdList.add(departmentId);
        DepartmentEntity departmentEntity = new DepartmentDTO();
        departmentEntity.setParentId(departmentId);
        QueryWrapper<DepartmentEntity> wrapper = new QueryWrapper<>(departmentEntity);
        List<DepartmentEntity> list = baseMapper.selectList(wrapper);
        for (DepartmentEntity entity : list) {
            findCurrentAndChildrenId(departmentIdList, entity.getId());
        }
    }

}
