package com.esv.datacenter.cc.module.tenant.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.esv.datacenter.cc.common.form.Add;
import com.esv.datacenter.cc.common.form.Update;
import com.esv.datacenter.cc.form.IdForm;
import com.esv.datacenter.cc.module.role.service.RoleService;
import com.esv.datacenter.cc.module.tenant.dto.TenantAdminRoleDTO;
import com.esv.datacenter.cc.module.tenant.entity.TenantEntity;
import com.esv.datacenter.cc.module.tenant.form.TenantForm;
import com.esv.datacenter.cc.module.tenant.service.TenantService;
import com.esv.datacenter.cc.module.tenant.vo.TenantVO;
import com.esv.common.response.R;
import com.esv.gateway.common.GatewayHeaders;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 公司表
 *
 * @author chenfm
 * @email chenfengman@esvtek.com
 * @date 2020-03-09 11:12:29
 */
@RestController
@RequestMapping("tenant")
public class TenantController {

    private TenantService tenantService;

    private RoleService roleService;

    @Autowired
    public TenantController(TenantService tenantService, RoleService roleService) {
        this.tenantService = tenantService;
        this.roleService = roleService;
    }

    /**
     * description 租户列表
     * param [tenantId, current, size, tenantName, status]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/18 10:36
     **/
    @GetMapping("list")
    public R list(
            @RequestHeader(GatewayHeaders.DATA_PERM) Integer dataPerm,
            @RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
            @RequestParam(value = "current", defaultValue = "1") Integer current,
            @RequestParam(value = "size", defaultValue = "10") Integer size,
            @RequestParam(value = "tenantName", required = false) String tenantName,
            @RequestParam(value = "status", required = false) Integer status
            ){
        IPage<TenantEntity> page = new Page<>(current, size);
        QueryWrapper<TenantEntity> wrapper = new QueryWrapper<>();
        if (dataPerm != 0) {
            wrapper.eq("id", tenantId);
        }
        if (StringUtils.isNotBlank(tenantName)) {
            wrapper.like("tenant_name", tenantName);
        }
        if (status != null) {
            wrapper.eq("status", status);
        }
        page = tenantService.page(page, wrapper);
        return R.ok(page);
    }

    /**
     * description 查询所有租户id和租户名
     * param [tenantId]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/12 17:41
     **/
    @GetMapping("all")
    public R all(
            @RequestHeader(GatewayHeaders.DATA_PERM) Integer dataPerm,
            @RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
            @RequestParam(value = "roleType", required = false) Integer roleType) {
        List<TenantVO> list = new ArrayList<>();
        if (dataPerm == 0) {
            // 全部数据权限,查询所有租户信息
            List<TenantAdminRoleDTO> tenantAdminRoleList = tenantService.tenantAdminRoleList(roleType);
            tenantAdminRoleList.forEach(tenantAdminRoleDTO -> {
                TenantVO tenantVO = new TenantVO(tenantAdminRoleDTO.getId(), tenantAdminRoleDTO.getTenantName());
                tenantVO.setAdminRoleId(tenantAdminRoleDTO.getAdminRoleId());
                list.add(tenantVO);
            });
        } else {
            // 租户或部门数据权限只查询当前租户
            TenantEntity tenantEntity = tenantService.getById(tenantId);
            TenantVO tenantVO = new TenantVO(tenantEntity.getId(), tenantEntity.getTenantName());
            tenantVO.setAdminRoleId(tenantEntity.getAdminRoleId());
            list.add(tenantVO);
        }
        return R.ok(list);
    }

    /**
     * description 保存租户信息
     * param [userId, tenantForm]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/18 10:36
     **/
    @PostMapping("add")
    public R add(@RequestHeader(GatewayHeaders.DATA_PERM) Integer dataPerm,
                 @RequestBody @Validated(Add.class) TenantForm tenantForm){
        if (dataPerm != 0) {
            return R.error("没有权限");
        }
        TenantEntity entity = tenantService.findByCode(tenantForm.getTenantCode());
        if (entity != null) {
            return R.error("租户编码已经存在");
        }

        // 新增租户
        tenantService.addTenant(tenantForm);

        return R.ok();
    }

    /**
     * description 修改租户信息
     * param [userId, tenantForm]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/18 10:36
     **/
    @PostMapping("update")
    public R update(@RequestBody @Validated(Update.class) TenantForm tenantForm){
        TenantEntity entity = tenantService.findByCode(tenantForm.getTenantCode());
        if ((entity != null) && (!entity.getId().equals(tenantForm.getId()))) {
            return R.error("租户编码已经存在");
        }
        TenantEntity tenantEntity = new TenantEntity();
        BeanUtils.copyProperties(tenantForm, tenantEntity);
		if (tenantService.updateById(tenantEntity)) {
            return R.ok();
        } else {
		    return R.error();
        }
    }

    /**
     * description 删除租户
     * param [idForm]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/18 10:36
     **/
    @PostMapping("delete")
    public R delete(@RequestBody IdForm idForm) {
        Long id = idForm.getId();
        if (id == 0) {
            return R.error("平台租户不可删除");
        }
        if (tenantService.canDelete(id)) {
            tenantService.deleteTenant(id);
            return R.ok();
        } else {
            return R.error("请先删除该租户下的所有信息");
        }
    }

}
