package com.esv.datacenter.cc.common.component;

import com.esv.datacenter.cc.util.AESSecretUtil;
import com.esv.datacenter.cc.module.account.bo.TokenBO;
import io.jsonwebtoken.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import javax.xml.bind.DatatypeConverter;
import java.security.Key;
import java.util.HashMap;
import java.util.Map;

/**
 * @description： JWT权限认证组件
 * @author：hcbmailbox@163.com
 * @date：2020/1/13
 */
@Component
@Slf4j
public class JwtAuthComponent {

    @Value("${jwt.generate.key:HelloWorld}")
    private String jwtGenerateKey;

    @Value("${jwt.secret.key:HelloWorld}")
    private String jwtSecretKey;

    /**
     * 生成JWT Token
     * @param playloadMap
     * @return
     */
    public String generateJWT(Map<String, Object> playloadMap) {
        JwtBuilder builder = Jwts.builder();

        // 设置JWT头
        Map<String, Object> headMap = new HashMap<>(2);
        headMap.put("alg", SignatureAlgorithm.HS256.getValue());
        headMap.put("typ", "JWT");
        builder.setHeader(headMap);

        // 设置有效载荷
        builder.addClaims(playloadMap);

        //签名算法，选择SHA-256
        SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS256;
        //将BASE64SECRET常量字符串使用base64解码成字节数组
        byte[] apiKeySecretBytes = DatatypeConverter.parseBase64Binary(jwtGenerateKey);
        //使用HmacSHA256签名算法生成一个HS256的签名秘钥Key
        Key signingKey = new SecretKeySpec(apiKeySecretBytes, signatureAlgorithm.getJcaName());

        // 设置签名
        builder.signWith(signatureAlgorithm, signingKey);

        return builder.compact();
    }

    /**
     * 创建Token
     * @param playloadMap
     * @return
     */
    public String generateToken(Map<String, Object> playloadMap) {
        // 创建JWT Token
        String jwtToken = this.generateJWT(playloadMap);

        // AES加密Token
        String token = AESSecretUtil.encryptToStr(jwtToken, jwtSecretKey);

        return token;
    }

    /**
     * 解析Token
     * @param jwtToken
     * @return
     */
    public Claims parseJWT(String jwtToken) {
        Claims claims = null;
        try {
            if (StringUtils.isNotBlank(jwtToken)) {
                //解析jwt
                JwtParser jwtParser = Jwts.parser().setSigningKey(DatatypeConverter.parseBase64Binary(jwtGenerateKey));
                claims = jwtParser.parseClaimsJws(jwtToken).getBody();
            } else {
                log.warn("jwtToken为空");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return claims;
    }

    /**
     * 获取Request请求Token
     * @param request
     * @return
     */
    public String getRequestToken(HttpServletRequest request) {
        String token = request.getHeader("Union-Authorization");
        if (StringUtils.isEmpty(token)) {
            return null;
        }
        if (!token.startsWith("Basic ")) {
            return null;
        }
        return StringUtils.trimToNull(token.replaceFirst("Basic ", ""));
    }

    /**
     * 判断请求Token是否有效
     * @param request
     * @return
     */
    public boolean isValidRequestToken(HttpServletRequest request) {
        String token = this.getRequestToken(request);
        if (null == token) {
            return false;
        }
        return isValidRequestToken(token);
    }

    /**
     * 判断请求Token是否有效
     * @param token
     * @return
     */
    public boolean isValidRequestToken(String token) {
        String jwtToken = AESSecretUtil.decryptToStr(token, jwtSecretKey);
        if (null == jwtToken) {
            return false;
        }
        Claims claims = this.parseJWT(jwtToken);
        if (null == claims) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * 解析Token数据
     * @param token
     * @return
     */
    public TokenBO parseToken(String token) {
        if (StringUtils.isEmpty(token)) {
            return null;
        }
        Claims claims;
        try {
            String jwtToken = AESSecretUtil.decryptToStr(token, jwtSecretKey);
            if (null == jwtToken) {
                return null;
            }
            claims = this.parseJWT(jwtToken);
            if (null == claims) {
                return null;
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
        TokenBO tokenBO = new TokenBO();
        tokenBO.setUserId(Long.parseLong(String.valueOf(claims.get("userId"))));
        tokenBO.setUserAccount((String) claims.get("userAccount"));
        tokenBO.setType(Integer.parseInt(String.valueOf(claims.get("type"))));
        tokenBO.setTenantId(Long.parseLong(String.valueOf(claims.get("tenantId"))));
        tokenBO.setDepartmentId(Long.parseLong(String.valueOf(claims.get("departmentId"))));
        tokenBO.setSource((String) claims.get("source"));
        tokenBO.setIp((String) claims.get("ip"));
        tokenBO.setCreateTime(Long.parseLong(String.valueOf(claims.get("createTime"))));
        tokenBO.setExpireTime(Long.parseLong(String.valueOf(claims.get("expireTime"))));
        tokenBO.setExtMap((Map<String, Object>) claims.get("extMap"));
        return tokenBO;
    }

    /**
     * 根据Token获取用户ID
     * @param token
     * @return
     */
    public Long getUserId(String token) {
        TokenBO tokenBO = this.parseToken(token);
        if (null != tokenBO) {
            return tokenBO.getUserId();
        } else {
            return null;
        }
    }

    /**
     * 根据Token获取用户帐号
     * @param token
     * @return
     */
    public String getUserAccount(String token) {
        TokenBO tokenBO = this.parseToken(token);
        if (null != tokenBO) {
            return tokenBO.getUserAccount();
        } else {
            return null;
        }
    }

}
