package com.esv.datacenter.cc.module.user.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.esv.datacenter.cc.common.constants.ApiResponseCode;
import com.esv.datacenter.cc.common.constants.Constant;
import com.esv.datacenter.cc.common.form.Add;
import com.esv.datacenter.cc.common.form.List;
import com.esv.datacenter.cc.common.form.Update;
import com.esv.datacenter.cc.module.user.dto.UserDTO;
import com.esv.datacenter.cc.module.user.entity.UserEntity;
import com.esv.datacenter.cc.module.user.form.UserForm;
import com.esv.datacenter.cc.module.user.service.UserRoleService;
import com.esv.datacenter.cc.module.user.service.UserService;
import com.esv.datacenter.cc.module.user.vo.UserInfoVO;
import com.esv.common.response.R;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.util.HashMap;
import java.util.Map;

/**
 * @description： 用户数据维护Controller
 * @author：hcbmailbox@163.com
 * @date：2020/1/10
 */
@RestController
@RequestMapping("user")
@Slf4j
@Validated
public class UserController {

    private UserService userService;

    private UserRoleService userRoleService;

    @Autowired
    public UserController(UserService userService, UserRoleService userRoleService) {
        this.userService = userService;
        this.userRoleService = userRoleService;
    }

    /**
     * 分页查询用户
     * @param form
     * @return
     * @throws Exception
     */
    @PostMapping("list")
    public R getUserList(@RequestBody @Validated(List.class) UserForm form) {
        int pageNum = Integer.parseInt(form.getCurrent());
        int pageSize = Integer.parseInt(form.getSize());
        IPage<UserDTO> page = new Page<>(pageNum, pageSize);

        Map<String, Object> queryObj = new HashMap<>(6);
        String departmentId = form.getDepartmentId();
        if (departmentId.startsWith("t_")) {
            // 选择的部门是租户
            queryObj.put("tenantId", Long.parseLong(departmentId.split("_")[1]));
            queryObj.put("departmentId", 0);
        } else {
            queryObj.put("departmentId", Long.parseLong(departmentId));
        }
        if (StringUtils.isNotBlank(form.getRealName())) {
            queryObj.put("realName", "%" + StringUtils.trim(form.getRealName()) + "%");
        }
        if (StringUtils.isNotBlank(form.getRoleId())) {
            queryObj.put("roleIds", StringUtils.trim(form.getRoleId()).split(","));
        }
        queryObj.put("deleted", false);
        userService.getUserList(page, queryObj);
        for (UserDTO userDTO : page.getRecords()) {
            if (!Constant.UserType.GENERAL.getValue().equals(userDTO.getType())) {
                userDTO.setCanEdited(false);
            }
        }
        return R.ok(page);
    }

    /**
     * 新增用户
     * @param form
     * @return
     * @throws Exception
     */
    @Transactional
    @PostMapping("add")
    public R add(@RequestBody @Validated(Add.class) UserForm form) {
        // 特殊账号不允许
        if ("admin".equalsIgnoreCase(form.getAccount()) || "root".equalsIgnoreCase(form.getAccount())) {
            return R.error(ApiResponseCode.PARAM_ERROR.code(), ApiResponseCode.PARAM_ERROR.message());
        }
        UserEntity entity = makeUserEntity(form);
        entity.setType(Constant.UserType.GENERAL.getValue());
        userService.insert(entity);

        // 保存用户角色
        userRoleService.editUserRole(entity.getId(), form.getRoleId());

        return R.ok();
    }

    /**
     * 更新用户信息
     * @param form
     * @return
     * @throws Exception
     */
    @Transactional
    @PostMapping("update")
    public R updateUserInfoById(@RequestBody @Validated(Update.class) UserForm form) {
        Long id = Long.parseLong(form.getId());
        UserEntity originUser = userService.getById(id);
        // 特殊账号不允许
        if (!originUser.getAccount().equals(form.getAccount())
            &&("admin".equalsIgnoreCase(form.getAccount()) || "root".equalsIgnoreCase(form.getAccount()))) {
            return R.error(ApiResponseCode.PARAM_ERROR.code(), ApiResponseCode.PARAM_ERROR.message());
        }
        UserEntity entity = makeUserEntity(form);
        entity.setId(id);
        entity.setAccount(null);
        userService.updateUserInfoById(entity);
        // 保存用户角色
        userRoleService.editUserRole(entity.getId(), form.getRoleId());

        return R.ok();
    }

    private UserEntity makeUserEntity(UserForm form) {
        UserEntity entity = new UserEntity();
        BeanUtils.copyProperties(form, entity);
        if (StringUtils.isNotBlank(form.getStatus())) {
            entity.setStatus(Integer.parseInt(form.getStatus()));
        }

        String departmentId = form.getDepartmentId();
        if (departmentId.startsWith("t_")) {
            // 租户
            entity.setTenantId(Long.parseLong(departmentId.split("_")[1]));
            entity.setDepartmentId(0L);
        } else {
            entity.setDepartmentId(Long.parseLong(departmentId));
        }
        return entity;
    }

    /**
     * 通过用户id获取用户信息
     * @param id
     * @return
     * @throws Exception
     */
    @GetMapping("get")
    public R getUserInfoById(@Pattern(regexp = "^[1-9]\\d*$", message = "无效的用户ID")
                                 @NotNull(message="用户ID不能为空")
                                 @NotBlank(message="用户ID不能为空")
                                 @RequestParam String id) {
        UserEntity userEntity = userService.getById(Long.parseLong(id));
        if (null == userEntity) {
            return R.error("无效的用户ID");
        }
        UserInfoVO userInfoVO = new UserInfoVO();
        BeanUtils.copyProperties(userEntity, userInfoVO);
        return R.ok(userInfoVO);
    }
}
