package com.esv.datacenter.cc.module.role.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.esv.datacenter.cc.common.constants.Constant;
import com.esv.datacenter.cc.common.form.Add;
import com.esv.datacenter.cc.common.form.Update;
import com.esv.datacenter.cc.form.IdForm;
import com.esv.datacenter.cc.module.department.entity.DepartmentEntity;
import com.esv.datacenter.cc.module.department.service.DepartmentService;
import com.esv.datacenter.cc.module.role.dto.RoleDTO;
import com.esv.datacenter.cc.module.role.dto.TenantRoleDTO;
import com.esv.datacenter.cc.module.role.entity.RoleEntity;
import com.esv.datacenter.cc.module.role.form.RoleForm;
import com.esv.datacenter.cc.module.role.service.RoleService;
import com.esv.datacenter.cc.module.tenant.service.TenantService;
import com.esv.datacenter.cc.module.user.service.UserRoleService;
import com.esv.common.response.R;
import com.esv.gateway.common.GatewayHeaders;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 角色
 *
 * @author chenfm
 * @email chenfengman@esvtek.com
 * @date 2020-01-09 11:32:08
 */
@Slf4j
@RefreshScope
@RestController
@RequestMapping("role")
public class RoleController {

    private RoleService roleService;

    private UserRoleService userRoleService;

    private TenantService tenantService;

    private DepartmentService departmentService;

    @Autowired
    public RoleController(RoleService roleService, UserRoleService userRoleService,
                          TenantService tenantService, DepartmentService departmentService) {
        this.roleService = roleService;
        this.userRoleService = userRoleService;
        this.tenantService = tenantService;
        this.departmentService = departmentService;
    }

    /**
     * @description 角色列表分页查询
     * @params [tenantId, current, size, categoryId, roleName]
     * @return com.esv.common.response.R
     * @author chenfm
     * @createTime 2020/3/11 10:05
     **/
    @GetMapping("list")
    public R list(
            @RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
            @RequestParam(value = "current", defaultValue = "1") Integer current,
            @RequestParam(value = "size", defaultValue = "10") Integer size,
            @RequestParam(value = "categoryId", required = false) Long categoryId,
            @RequestParam(value = "roleName", required = false) String roleName,
            @RequestParam(value = "tenantId", required = false) Long queryTenantId) {
        RoleDTO roleDTO = new RoleDTO();
        if (queryTenantId == null) {
            if (tenantId != 0) {
                roleDTO.setTenantId(tenantId);
            }
        } else {
            roleDTO.setTenantId(queryTenantId);
        }
        roleDTO.setCategoryId(categoryId);
        roleDTO.setRoleName(roleName);
        roleDTO.setType(Constant.RoleType.GENERAL.getValue());
        IPage<RoleDTO> page = new Page<>(current, size);
        roleService.listRoleDTOPage(page, roleDTO);
        for (RoleDTO dto : page.getRecords()) {
            boolean roleCanEdited = roleCanEdited(dto.getId());
            dto.setCanEdited(roleCanEdited);
        }
        return R.ok(page);
    }

    private boolean roleCanEdited(Long roleId) {
        if (roleService.isSuperAdminRole(roleId) || tenantService.isTenantAdminRole(roleId)) {
            return false;
        }
        return true;
    }

    /**
     * description 查询所有角色
     * param [tenantId]
     * return com.esv.common.response.R
     * author chenfm
     * createTime 2020/3/16 10:29
     **/
    @GetMapping("all")
    public R all(@RequestParam(value = "type", required = false) Integer type,
                 @RequestParam(value = "departmentId", required = false) String departmentId) {
        Long tenantId = null;
        if (StringUtils.isNotBlank(departmentId)) {
            if (departmentId.startsWith("t_")) {
                tenantId = Long.parseLong(departmentId.split("_")[1]);
            } else {
                DepartmentEntity entity = departmentService.getById(departmentId);
                tenantId = entity.getTenantId();
            }
        }
        List<TenantRoleDTO> tenantRoleDTOList = roleService.allByTenant(tenantId, type);
        return R.ok(tenantRoleDTOList);
    }

    /**
     * @description 新增角色
     * param [tenantId, userId, roleEntity]
     * @return com.esv.common.response.R
     * @author chenfm
     * @createTime 2020/3/11 10:05
     **/
    @PostMapping("add")
    public R add(
            @RequestHeader(GatewayHeaders.TENANT_ID) Long tenantId,
            @RequestHeader(GatewayHeaders.DEPARTMENT_ID) Long departmentId,
            @RequestBody @Validated(Add.class) RoleForm roleForm) {
        RoleEntity roleEntity = new RoleEntity();
        BeanUtils.copyProperties(roleForm, roleEntity);
        // 添加本租户内角色记录部门id, 添加其他租户角色将部门id记为0
        if (roleForm.getTenantId().equals(tenantId)) {
            roleEntity.setDepartmentId(departmentId);
        } else {
            roleEntity.setDepartmentId(0L);
        }
        roleEntity.setType(Constant.RoleType.GENERAL.getValue());
        if (roleService.save(roleEntity)) {
            return R.ok();
        } else {
            return R.error();
        }
    }

    /**
     * @description 修改角色
     * param [userId, roleEntity]
     * @return com.esv.common.response.R
     * @author chenfm
     * @createTime 2020/3/11 10:05
     **/
    @PostMapping("update")
    public R update(@RequestBody @Validated(Update.class) RoleForm roleForm) {
        RoleEntity roleEntity = new RoleEntity();
        BeanUtils.copyProperties(roleForm, roleEntity);
        if (roleService.updateById(roleEntity)) {
            return R.ok();
        } else {
            return R.error();
        }
    }

    /**
     * @description 删除角色
     * @param idForm
     * @return com.esv.common.response.R
     * @author chenfm
     * @createTime 2020/3/11 10:05
     **/
    @PostMapping("delete")
    public R delete(@RequestBody IdForm idForm) {
        long id = idForm.getId();
        if (!roleCanEdited(id)) {
            return R.error("平台管理员和租户管理员角色不可删除");
        }
        if (userRoleService.countByRoleId(id) > 0) {
            return R.error("不可删除, 请删除该角色下所有权限和用户");
        }
        roleService.removeById(idForm.getId());
        return R.ok();
    }

}
