package com.esv.datacenter.cc.module.authentication.feign;

import com.esv.common.response.ECode;
import com.esv.common.response.EResponse;
import com.esv.datacenter.cc.common.component.JwtAuthComponent;
import com.esv.datacenter.cc.common.constants.ApiResponseCode;
import com.esv.datacenter.cc.module.account.bo.TokenBO;
import com.esv.datacenter.cc.module.apiauth.service.ApiAuthorityService;
import com.esv.datacenter.cc.module.authentication.service.AuthenticationService;
import com.esv.datacenter.cc.util.RequestUriUtils;
import com.esv.feign.cc.client.AuthFeignClient;
import com.esv.feign.cc.req.WebReqAuthReq;
import com.esv.feign.cc.res.WebReqAuthRes;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;

/**
 * @description:
 * @project: cc
 * @name: AuthFeignClientImpl
 * @author: chenfm
 * @email: chenfengman@esvtek.com
 * @createTime: 2020/3/6 15:48
 * @version:1.0
 */
@Slf4j
@RestController
public class AuthFeignClientImpl implements AuthFeignClient {

    @Value("${auth.api.check.switch:true}")
    private String apiCheckSwitch;

    /**
     * 文件下载url
     **/
    @Value("${auth.api.file-download-url:/file/download/direct}")
    private String fileDownloadUrl;

    private ApiAuthorityService apiAuthorityService;

    private JwtAuthComponent jwtAuthComponent;

    private AuthenticationService authenticationService;

    @Autowired
    public AuthFeignClientImpl(ApiAuthorityService apiAuthorityService, JwtAuthComponent jwtAuthComponent,
                               AuthenticationService authenticationService) {
        this.apiAuthorityService = apiAuthorityService;
        this.jwtAuthComponent = jwtAuthComponent;
        this.authenticationService = authenticationService;
    }

    @Override
    public EResponse<WebReqAuthRes> webReqAuth(String traceId, WebReqAuthReq webReqAuthReq) {
        String reqUrl = webReqAuthReq.getReqUrl();
        if (reqUrl.startsWith(fileDownloadUrl)) {
            reqUrl = fileDownloadUrl;
        }
        String token = webReqAuthReq.getToken();
        String menuCode = webReqAuthReq.getMenuCode();
        if (log.isDebugEnabled()) {
            log.debug("系统鉴权开始 -- menuCode:{}, reqUrl:{}, token:{}", menuCode, reqUrl, token);
        }

        WebReqAuthRes webReqAuthRes = new WebReqAuthRes();
        webReqAuthRes.setSystemCode(RequestUriUtils.getSystemCode(reqUrl));
        webReqAuthRes.setDataPerm(0);
        Map<String, Object> apiMap = apiAuthorityService.getAllApiMap();
        if (!apiMap.containsKey(reqUrl)) {
            // URL未注册
            if (log.isDebugEnabled()) {
                log.debug("URL未注册:{}", reqUrl);
            }
            return EResponse.error(ApiResponseCode.URL_REGISTERED);
        }
        // 鉴权类型
        int authType = (int) apiMap.get(reqUrl);
        if (log.isDebugEnabled()) {
            log.debug("该接口鉴权类型authType:{}", authType);
        }
        // 接口不需鉴权
        if (0 == authType) {
            if (log.isDebugEnabled()) {
                log.debug("该请求不需要鉴权");
            }
            return EResponse.ok(webReqAuthRes).message("该请求不需要鉴权");
        }
        // 判断token格式是否正确
        if (StringUtils.isEmpty(token)) {
            if (log.isDebugEnabled()) {
                log.debug("无效的Token");
            }
            return EResponse.error(ECode.TOKEN_INVALID);
        } else {
            if (!token.startsWith("Basic ")) {
                // token无效
                if (log.isDebugEnabled()) {
                    log.debug("Token格式错误");
                }
                return  EResponse.error(ECode.TOKEN_INVALID);
            }
            token = StringUtils.trimToNull(token.replaceFirst("Basic ", ""));
        }
        // 解析token
        TokenBO tokenBO = jwtAuthComponent.parseToken(token);
        if (null == tokenBO) {
            if (log.isDebugEnabled()) {
                log.debug("无效的Token");
            }
            return EResponse.error(ECode.TOKEN_INVALID);
        } else if ((0L != tokenBO.getExpireTime())
                && (System.currentTimeMillis() > tokenBO.getExpireTime())) {
            if (log.isDebugEnabled()) {
                log.debug("Token已过期");
            }
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }
        // 拼装返回信息
        webReqAuthRes.setUserId(tokenBO.getUserId());
        webReqAuthRes.setAccount(tokenBO.getUserAccount());
        webReqAuthRes.setTenantId(tokenBO.getTenantId());
        webReqAuthRes.setDepartmentId(tokenBO.getDepartmentId());
        // 接口只需token, 不需要具体权限
        if (1 == authType) {
            if (log.isDebugEnabled()) {
                log.debug("鉴权通过");
            }
            // 设置数据权限
            authenticationService.setTokenDataPerm(tokenBO, webReqAuthRes);
            return EResponse.ok(webReqAuthRes);
        }

        // 接口需要具体权限
        if ("true".equals(apiCheckSwitch)) {
            return authenticationService.access(webReqAuthRes, menuCode, reqUrl);
        } else {
            webReqAuthRes.setDataPerm(1);
            return EResponse.ok(webReqAuthRes);
        }
    }

}
