package com.esv.superhive.base.module.captcha.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.superhive.base.common.response.ECode;
import com.esv.superhive.base.common.util.AESSecretUtils;
import lombok.extern.slf4j.Slf4j;
import org.junit.*;
import org.junit.runner.RunWith;
import org.junit.runners.MethodSorters;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultHandlers;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.context.WebApplicationContext;

/**
 * @description:
 * @project: base-service
 * @name: com.esv.htwl.base.module.captcha.controller.CaptchaControllerTest
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/11 11:34
 * @version:1.0
 */
@RunWith(SpringRunner.class)
@SpringBootTest
@Slf4j
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class CaptchaControllerTest {

    @Value("${aes.sha1prng.key:3.1415926535}")
    private String AES_KEY;

    @Autowired
    private WebApplicationContext webApplicationContext;

    private MockMvc mockMvc;

    private static String captchaID;

    @Before
    public void before() {
        log.info("=================================== Test Start ===================================");
        mockMvc = MockMvcBuilders.webAppContextSetup(webApplicationContext).build();
    }

    @After
    public void after() {
        log.info("=================================== Test End ===================================");
    }

    @Test
    public void a1_getDefaultCaptcha_success_test() throws Exception {
        String url = "/captcha/get";
        JSONObject reqJson = new JSONObject();

        MvcResult mvcResult = mockMvc.perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));

        this.captchaID = result.getJSONObject("data").getString("id");
    }

    @Test
    public void b1_verifyCaptcha_success_test() throws Exception {
        String url = "/captcha/verify";
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", captchaID);
        reqJson.put("captcha", AESSecretUtils.decryptToStr(captchaID, AES_KEY).split(",")[1]);

        MvcResult mvcResult = mockMvc.perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(result.getJSONObject("data").getBooleanValue("result"));
    }

    @Test
    public void b2_verifyCaptcha_params_all_error_test() throws Exception {
        String url = "/captcha/verify";
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", "error_id_test");
        reqJson.put("captcha", "12_3");

        MvcResult mvcResult = mockMvc.perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertFalse(result.getJSONObject("data").getBooleanValue("result"));
    }

    @Test
    public void b3_verifyCaptcha_params_id_error_test() throws Exception {
        String url = "/captcha/verify";
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", "error_id");
        reqJson.put("captcha", AESSecretUtils.decryptToStr(captchaID, AES_KEY).split(",")[1]);

        MvcResult mvcResult = mockMvc.perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertFalse(result.getJSONObject("data").getBooleanValue("result"));
    }

    @Test
    public void b4_verifyCaptcha_params_captcha_error_test() throws Exception {
        String url = "/captcha/verify";
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", captchaID);
        reqJson.put("captcha", "12_3");

        MvcResult mvcResult = mockMvc.perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertFalse(result.getJSONObject("data").getBooleanValue("result"));
    }
}
