package com.esv.freight.app.module.address.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.common.validator.groups.ValidatorDetail;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.common.validator.groups.ValidatorList;
import com.esv.freight.app.feign.ReceiveAddressInterface;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.address.form.AddressQueryForm;
import com.esv.freight.app.module.address.form.ReceiveAddressForm;
import com.esv.freight.app.module.address.vo.AddressDetailVO;
import com.esv.freight.app.module.address.vo.AddressBriefVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 收货地址Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.address.controller.ReceiveAddressController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/13 10:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/address/receive")
@Slf4j
@Validated
public class ReceiveAddressController {

    private AppLoginService appLoginService;
    private ReceiveAddressInterface receiveAddressInterface;

    @Autowired
    public ReceiveAddressController(ReceiveAddressInterface receiveAddressInterface, AppLoginService appLoginService) {
        this.appLoginService = appLoginService;
        this.receiveAddressInterface = receiveAddressInterface;
    }

    /**
     * description 收货地址列表
     * param [accessToken]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/17 13:00
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody(required=false) @Validated(ValidatorList.class) AddressQueryForm addressQueryForm) {

        appLoginService.checkAccessToken();

        // 调用获取货主发货地址列表接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("ownerId", addressQueryForm.getOwnerId());
        JSONObject result = receiveAddressInterface.all(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        List<AddressBriefVO> addressBriefVOList = new ArrayList<>();
        JSONArray items = result.getJSONArray("data");
        if(items.size() > 0) {
            for(int i=0; i<items.size(); ++i) {
                AddressBriefVO vo = new AddressBriefVO();
                JSONObject object = items.getJSONObject(i);
                vo.setId(object.getLong("id"));
                vo.setAddressName(object.getString("addressName"));
                vo.setProvinceCode(object.getString("provinceCode"));
                vo.setCityCode(object.getString("cityCode"));
                vo.setDistrictCode(object.getString("districtCode"));
                vo.setDetailAddress(object.getString("detailAddress"));
                addressBriefVOList.add(vo);
            }
        }
        return EResponse.ok(addressBriefVOList);
    }

    /**
     * description 新增收货地址
     * param [CreateAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody(required=false) @Validated(ValidatorInsert.class) ReceiveAddressForm receiveAddressForm) {

        appLoginService.checkAccessToken();

        // 调用添加货主发货地址接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("ownerId", receiveAddressForm.getOwnerId());
        reqJson.put("addressName", receiveAddressForm.getAddressName());
        reqJson.put("provinceCode", receiveAddressForm.getProvinceCode());
        reqJson.put("cityCode", receiveAddressForm.getCityCode());
        reqJson.put("districtCode", receiveAddressForm.getDistrictCode());
        reqJson.put("detailAddress", receiveAddressForm.getDetailAddress());
        reqJson.put("receiver", receiveAddressForm.getReceiver());
        reqJson.put("receiverPhone", receiveAddressForm.getReceiverPhone());
        log.info(reqJson.toJSONString());
        JSONObject result = receiveAddressInterface.add(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 编辑收货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody(required=false) @Validated(ValidatorInsert.class) ReceiveAddressForm receiveAddressForm) {

        appLoginService.checkAccessToken();

        // 调用编辑货主发货地址接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", receiveAddressForm.getId());
        reqJson.put("ownerId", receiveAddressForm.getOwnerId());
        reqJson.put("addressName", receiveAddressForm.getAddressName());
        reqJson.put("provinceCode", receiveAddressForm.getProvinceCode());
        reqJson.put("cityCode", receiveAddressForm.getCityCode());
        reqJson.put("districtCode", receiveAddressForm.getDistrictCode());
        reqJson.put("detailAddress", receiveAddressForm.getDetailAddress());
        reqJson.put("receiver", receiveAddressForm.getReceiver());
        reqJson.put("receiverPhone", receiveAddressForm.getReceiverPhone());
        JSONObject result = receiveAddressInterface.edit(reqJson);

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 删除收货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody(required=false) @Validated(ValidatorInsert.class) AddressQueryForm addressQueryForm) {

        appLoginService.checkAccessToken();

        // 调用获取删除货主发货地址接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", addressQueryForm.getId());
        JSONObject result = receiveAddressInterface.delete(reqJson);

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description  收货地址详情
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/23 11:00
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody(required=false) @Validated(ValidatorDetail.class) ReceiveAddressForm receiveAddressForm) {

        appLoginService.checkAccessToken();

        // 调用获取发货地址详情接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", receiveAddressForm.getId());
        JSONObject result = receiveAddressInterface.detail(reqJson);

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        AddressDetailVO addressDetailVO = new AddressDetailVO();
        JSONObject data = result.getJSONObject("data");
        addressDetailVO.setAddressName(data.getString("addressName"));
        addressDetailVO.setProvinceCode(data.getString("provinceCode"));
        addressDetailVO.setCityCode(data.getString("cityCode"));
        addressDetailVO.setDistrictCode(data.getString("districtCode"));
        addressDetailVO.setDetailAddress(data.getString("detailAddress"));
        addressDetailVO.setContact(data.getString("receiver"));
        addressDetailVO.setContactPhone(data.getString("receiverPhone"));
        return EResponse.ok(addressDetailVO);
    }
}
