package com.esv.freight.app.module.message.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.app.common.constants.AccountConstants;
import com.esv.freight.app.common.exception.EException;
import com.esv.freight.app.common.vo.PageResultVO;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.entity.AppLoginEntity;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.message.dao.BusinessMessageDao;
import com.esv.freight.app.module.message.entity.BusinessMessageEntity;
import com.esv.freight.app.module.message.form.MessageForm;
import com.esv.freight.app.module.message.form.PushMessageForm;
import com.esv.freight.app.module.message.service.AccountDeviceInfoService;
import com.esv.freight.app.module.message.service.BusinessMessageService;
import com.esv.freight.app.module.message.vo.MessageDetailVO;
import com.esv.freight.app.module.message.vo.MessageListItemVO;
import com.esv.freight.app.module.message.vo.MessageTypeVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Slf4j
@Service("BusinessMessageService")
public class BusinessMessageServiceImpl extends ServiceImpl<BusinessMessageDao, BusinessMessageEntity> implements BusinessMessageService {

    private NoticeInterface noticeInterface;
    private AccountDeviceInfoService accountDeviceInfoService;
    private AppLoginService appLoginService;

    @Autowired
    public BusinessMessageServiceImpl(AppLoginService appLoginService, NoticeInterface noticeInterface, AccountDeviceInfoService accountDeviceInfoService) {
        this.noticeInterface = noticeInterface;
        this.accountDeviceInfoService = accountDeviceInfoService;
        this.appLoginService = appLoginService;
    }

    @Override
    public void insertMessage(PushMessageForm pushMessageForm) {

        // 1:保存消息
        BusinessMessageEntity entity = new BusinessMessageEntity();
        BeanUtils.copyProperties(pushMessageForm, entity);
        entity.setAccountType(pushMessageForm.getAppType());

        JSONObject jo;
        if(pushMessageForm.getMessageType() == AccountConstants.MESSAGE_TYPE_BILL_CONFIRM) {
            jo = get101Message(pushMessageForm);
        }
        else if(pushMessageForm.getMessageType() == AccountConstants.MESSAGE_TYPE_ACCOUNT_NOTICE) {
            jo = get102Message(pushMessageForm);
        }
        else {
            throw new EException(1001, "无效的消息类型");
        }

        entity.setMessageContent(jo.getString("msg_content"));
        entity.setMessageTitle(jo.getString("title"));
        entity.setMessageData(pushMessageForm.getMessageData());
        save(entity);

        // 2:调用给指定设备推送消息接口
        AppLoginEntity appLoginEntity = appLoginService.getAccountById(pushMessageForm.getAccountId());
        if(appLoginEntity == null) {
            return;
        }

        JSONObject reqJson = new JSONObject();
        if(pushMessageForm.getAppType() == AccountConstants.ACCOUNT_TYPE_DRIVER) {
            reqJson.put("appType", 2); // 给指定设备推送消息接口APP类型：1-货主、2-司机
        }
        else {
            reqJson.put("appType", 1);
        }

        if(appLoginEntity.getDeviceType() == AccountConstants.APP_TYPE_ANDROID) {
            reqJson.put("deviceType", 1); // 设备类型：1-android、2-ios
        }
        else {
            reqJson.put("deviceType", 2);
        }

        reqJson.put("registrationId", accountDeviceInfoService.getDeviceSn(pushMessageForm.getAccountId(), appLoginEntity.getDeviceType()));
        reqJson.put("message", jo);

        JSONObject result;
        log.info(reqJson.toJSONString());
        result = noticeInterface.push2Device(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") == 200) {
            String msgId = result.getJSONObject("data").getString("msgId");
            lambdaUpdate()
                    .eq(BusinessMessageEntity::getId, entity.getId())
                    .set(BusinessMessageEntity::getMsgId, msgId)
                    .update();
        }
    }

    private JSONObject get101Message(PushMessageForm pushMessageForm) {
        JSONObject jo = JSONObject.parseObject(pushMessageForm.getMessageData());
        String billNo = jo.getString("billNo");
        JSONObject reqMessage = new JSONObject();
        reqMessage.put("title", "账单提醒");
        String content = "您的新账单 "+billNo+" 已经生成，请尽快确认";
        reqMessage.put("msg_content", content);
        JSONObject extras = new JSONObject();
        extras.put("messageType", AccountConstants.MESSAGE_TYPE_BILL_CONFIRM);
        extras.put("data", jo);
        reqMessage.put("extras", extras);
        return reqMessage;
    }

    private JSONObject get102Message(PushMessageForm pushMessageForm) {
        JSONObject jo = JSONObject.parseObject(pushMessageForm.getMessageData());
        Integer resultCode = jo.getInteger("resultCode");
        JSONObject reqMessage = new JSONObject();
        reqMessage.put("title", "账号提醒");
        if(resultCode == 1) {
            if(pushMessageForm.getAppType() == AccountConstants.ACCOUNT_TYPE_DRIVER) {
                reqMessage.put("msg_content", "您的账号已审核通过了，赶快去添加车辆吧");
            }
            else if(pushMessageForm.getAppType() == AccountConstants.ACCOUNT_TYPE_GOODS_OWNER) {
                reqMessage.put("msg_content", "您的账号已审核通过了，赶快去发布货源吧");
            }
        }
        else {
            String resultMessage = jo.getString("resultMessage");
            String content = "您的账号已审核失败了";
            if(!StringUtils.isEmpty(resultMessage)) {
                content = content + "，" + resultMessage;
            }

            reqMessage.put("msg_content", content);
        }

        JSONObject extras = new JSONObject();
        extras.put("messageType", AccountConstants.MESSAGE_TYPE_ACCOUNT_NOTICE);
        extras.put("data", jo);
        reqMessage.put("extras", extras);
        return reqMessage;
    }

    private MessageTypeVO getLastestMessageType(Integer messageType, Long accountId) {
        List<BusinessMessageEntity> lstMessage = lambdaQuery()
                .eq(BusinessMessageEntity::getAccountId, accountId)
                .eq(BusinessMessageEntity::getMessageType, messageType)
                .orderByDesc(BusinessMessageEntity::getCreateTime).list();

        if(lstMessage != null && lstMessage.size() > 0) {
            BusinessMessageEntity businessMessageEntity = lstMessage.get(0);
            MessageTypeVO vo = new MessageTypeVO();
            BeanUtils.copyProperties(businessMessageEntity, vo);

            boolean hasNoRead = false;
            for(BusinessMessageEntity entity : lstMessage) {
                if(entity.getIsRead() == false) {
                    hasNoRead = true;
                    break;
                }
            }

            vo.setIsRead(!hasNoRead);
            return vo;
        }

        return null;
    }

    @Override
    public List<MessageTypeVO> getMessageType(MessageForm messageForm) {
        // 1.分类查询消息，每个type类型消息只取最后生成的一个消息
        List<MessageTypeVO> messageTypeVOList = new ArrayList<>();

        MessageTypeVO systemMessage = new MessageTypeVO();
        systemMessage.setIsRead(true);
        systemMessage.setMessageType(AccountConstants.MESSAGE_TYPE_SYSTEM);
        systemMessage.setMessageContent(null);
        systemMessage.setMessageTitle(null);
        systemMessage.setCreateTime(null);
        messageTypeVOList.add(systemMessage);

        MessageTypeVO msg101 = getLastestMessageType(AccountConstants.MESSAGE_TYPE_BILL_CONFIRM, messageForm.getAccountId());
        if(msg101 != null) {
            messageTypeVOList.add(msg101);
        }

        MessageTypeVO msg102 = getLastestMessageType(AccountConstants.MESSAGE_TYPE_ACCOUNT_NOTICE, messageForm.getAccountId());
        if(msg102 != null) {
            messageTypeVOList.add(msg102);
        }

        return messageTypeVOList;
    }

    @Override
    public PageResultVO getMessageList(MessageForm messageForm) {

        // 分页查询
        IPage<BusinessMessageEntity> page = new Page<>(messageForm.getPageNum(), messageForm.getPageSize());
        IPage<BusinessMessageEntity> entityPage = lambdaQuery()
                .eq(BusinessMessageEntity::getAccountId, messageForm.getAccountId())
                .eq(BusinessMessageEntity::getMessageType, messageForm.getMessageType())
                .orderByDesc(BusinessMessageEntity::getCreateTime)
                .page(page);

        // 数据转换
        List<MessageListItemVO> list = new ArrayList<>();
        for(BusinessMessageEntity entity: entityPage.getRecords()){
            MessageListItemVO itemVO = new MessageListItemVO();
            BeanUtils.copyProperties(entity, itemVO);
            list.add(itemVO);
        }

        PageResultVO pageResultVO = new PageResultVO(page, list);

        lambdaUpdate()
                .eq(BusinessMessageEntity::getAccountId, messageForm.getAccountId())
                .eq(BusinessMessageEntity::getMessageType, messageForm.getMessageType())
                .set(BusinessMessageEntity::getIsRead, 1)
                .update();

        return pageResultVO;
    }

    @Override
    public void deleteMessageById(MessageForm messageForm) {
        QueryWrapper<BusinessMessageEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(BusinessMessageEntity::getId, messageForm.getId());
        this.remove(queryWrapper);
    }

    @Override
    public void deleteMessageByType(MessageForm messageForm) {
        QueryWrapper<BusinessMessageEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(BusinessMessageEntity::getMessageType, messageForm.getMessageType())
                .eq(BusinessMessageEntity::getAccountId, messageForm.getAccountId());
        this.remove(queryWrapper);
    }

    @Override
    public void setAllRead(MessageForm messageForm) {
        lambdaUpdate()
                .eq(BusinessMessageEntity::getAccountId, messageForm.getAccountId())
                .set(BusinessMessageEntity::getIsRead, 1)
                .update();
    }

    @Override
    public MessageDetailVO getMessageDetail(MessageForm messageForm) {

        BusinessMessageEntity entity = lambdaQuery()
                .eq(BusinessMessageEntity::getId, messageForm.getId())
                .one();

        MessageDetailVO messageDetailVO = new MessageDetailVO();
        BeanUtils.copyProperties(entity, messageDetailVO);
        return messageDetailVO;
    }

    @Override
    public Integer hasUnReadMessage(Long accountId) {
        List<BusinessMessageEntity> entityList = lambdaQuery()
                .eq(BusinessMessageEntity::getAccountId, accountId)
                .eq(BusinessMessageEntity::getIsRead, 0)
                .list();

        if(entityList == null || entityList.size() == 0) {
            return 0;
        }

        return 1;
    }
}
