package com.esv.freight.app.module.account.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.feign.DictInterface;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.form.LoginForm;
import com.esv.freight.app.module.account.form.RefreshTokenForm;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginByPwd;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginBySms;
import com.esv.freight.app.module.account.vo.DriverAccountDetailVO;
import com.esv.freight.app.module.account.vo.LoginVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * @description: 账号Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.account.controller.DriverAccountController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/14 14:00
 * @version:1.0
 */
@RestController
@RequestMapping("/driverBackend/account")
@Slf4j
@Validated
public class DriverAccountController {

    private DictInterface dictInterface;
    private NoticeInterface noticeInterface;
    private AppLoginService appLoginService;

    @Autowired
    public DriverAccountController(DictInterface dictInterface, NoticeInterface noticeInterface, AppLoginService appLoginService) {
        this.dictInterface = dictInterface;
        this.noticeInterface = noticeInterface;
        this.appLoginService = appLoginService;
    }

    /**
     * description 短信验证码登录
     * param [LoginBySmsForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/login/loginBySms")
    public EResponse loginBySms(@RequestHeader @RequestBody(required=false) @Validated(ValidatorLoginBySms.class) LoginForm loginForm) {

        // 调用通知服务验证短信接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", loginForm.getPhone());
        reqJson.put("type", "login");
        reqJson.put("captcha", loginForm.getSmsCode());
        log.info(reqJson.toJSONString());
        JSONObject resultCheck = noticeInterface.checkSmsCaptcha(reqJson);
        log.info(resultCheck.toJSONString());

        if(resultCheck.getInteger("code") != 200) {
            return EResponse.error(resultCheck.getInteger("code"), resultCheck.getString("message"));
        }

        // todo 调用注册帐号接口，等待司机管理服务接口

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号密码登录RequestHeader
     * param [LoginByPwdForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/login/loginByPwd")
    public EResponse loginByPwd(@RequestBody(required=false) @Validated(ValidatorLoginByPwd.class) LoginForm loginForm) {

        // todo 调用帐号密码校验接口，等待司机管理服务接口

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号登出
     * param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/logout")
    public EResponse logout() {
        String accessToken = ReqUtils.getTokenInfo().getAccessToken();
        appLoginService.logout(accessToken);
        return EResponse.ok();
    }

    /**
     * description 刷新Token
     * param [refreshToken] 刷新Token
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 09:00
     **/
    @PostMapping("/token/refresh")
    public EResponse refresh(@RequestBody(required=false) @Validated(ValidatorInsert.class) RefreshTokenForm refreshTokenForm) {

        String accessToken = ReqUtils.getTokenInfo().getAccessToken();
        LoginVO loginByPwdVO = appLoginService.refreshToken(accessToken, refreshTokenForm);
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 获取帐号详情
     * param []
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 15:00
     **/
    @PostMapping("/detail")
    public EResponse detail() {
        String accessToken = ReqUtils.getTokenInfo().getAccessToken();
        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        DriverAccountDetailVO detailVO = new DriverAccountDetailVO();
        detailVO.setId("12345678");
        detailVO.setAccount("15040196726");
        detailVO.setAuditStatus(1);
        detailVO.setSourceType("2");
        detailVO.setName("测试666");
        detailVO.setIdCard("210112198707070023");
        detailVO.setIdCardExpireDate("2030.02.05");
        detailVO.setIdCardFileBackId("111");
        detailVO.setIdCardFileFrontId("222");
        detailVO.setProvinceCode("210000");
        detailVO.setCityCode("210100");
        detailVO.setDistrictCode("210104");
        detailVO.setDetailAddress("惠工街206号卓展中心");
        return EResponse.ok(detailVO);
    }
}
