package com.esv.freight.app.module.account.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.feign.DictInterface;
import com.esv.freight.app.feign.GoodsOwnerInterface;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.form.*;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginByPwd;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginBySms;
import com.esv.freight.app.module.account.vo.OwnerAccountDetailVO;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.module.account.vo.LoginVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * @description: 账号Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.account.controller.OwnerAccountController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/09 17:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/account")
@Slf4j
@Validated
public class OwnerAccountController {

    private DictInterface dictInterface;
    private NoticeInterface noticeInterface;
    private GoodsOwnerInterface goodsOwnerInterface;
    private AppLoginService appLoginService;

    @Autowired
    public OwnerAccountController(GoodsOwnerInterface goodsOwnerInterface, DictInterface dictInterface, NoticeInterface noticeInterface, AppLoginService appLoginService) {
        this.dictInterface = dictInterface;
        this.goodsOwnerInterface = goodsOwnerInterface;
        this.noticeInterface = noticeInterface;
        this.appLoginService = appLoginService;
    }

    /**
     * description 短信验证码登录
     * param [LoginBySmsForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/10 16:00
     **/
    @PostMapping("/login/loginBySms")
    public EResponse loginBySms(@RequestBody(required=false) @Validated(ValidatorLoginBySms.class) LoginForm loginForm) {

        // 调用通知服务验证短信接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", loginForm.getPhone());
        reqJson.put("type", "login");
        reqJson.put("captcha", loginForm.getSmsCode());
        log.info(reqJson.toJSONString());
        JSONObject resultCheck = noticeInterface.checkSmsCaptcha(reqJson);
        log.info(resultCheck.toJSONString());

        if(resultCheck.getInteger("code") != 200) {
            return EResponse.error(resultCheck.getInteger("code"), resultCheck.getString("message"));
        }

        // 调用注册帐号接口
        reqJson.clear();
        reqJson.put("account", loginForm.getPhone());
        log.info(reqJson.toJSONString());
        JSONObject resultRegister = goodsOwnerInterface.accountRegister(reqJson);
        log.info(resultRegister.toJSONString());

        // 1001表示 帐号已存在
        if(resultRegister.getInteger("code") != 200 && resultRegister.getInteger("code") != 1001) {
            return EResponse.error(resultRegister.getInteger("code"), resultRegister.getString("message"));
        }

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号密码登录RequestHeader
     * param [LoginByPwdForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/10 16:00
     **/
    @PostMapping("/login/loginByPwd")
    public EResponse loginByPwd(@RequestBody(required=false) @Validated(ValidatorLoginByPwd.class) LoginForm loginForm) {

        // 调用帐号密码校验接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("account", loginForm.getPhone());
        reqJson.put("password", loginForm.getPwd());
        log.info(reqJson.toJSONString());
        JSONObject result = goodsOwnerInterface.checkAccountPwd(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号登出
     * param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/11 11:00
     **/
    @PostMapping("/logout")
    public EResponse logout(@RequestHeader("Union-Authorization") String accessToken) {
        appLoginService.logout(accessToken);
        return EResponse.ok();
    }

    /**
     * description 刷新Token
     * param [refreshToken] 刷新Token
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 09:00
     **/
    @PostMapping("/token/refresh")
    public EResponse refresh(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) RefreshTokenForm refreshTokenForm) {

        LoginVO loginByPwdVO = appLoginService.refreshToken(accessToken, refreshTokenForm);
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 获取帐号详情
     * param []
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestHeader("Union-Authorization") String accessToken) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        String phone = appLoginService.getPhoneByAccessToken(accessToken);

        // 调用帐号密码校验接口
        JSONObject reqJsonDetail = new JSONObject();
        reqJsonDetail.put("account", phone);
        log.info(reqJsonDetail.toJSONString());
        JSONObject result = goodsOwnerInterface.getAccountDetail(reqJsonDetail);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OwnerAccountDetailVO accountDetailVO = new OwnerAccountDetailVO();
        accountDetailVO.setId(String.valueOf(result.getJSONObject("data").getLong("id")));
        accountDetailVO.setAccount(result.getJSONObject("data").getString("account"));
        accountDetailVO.setSourceType(result.getJSONObject("data").getString("sourceType"));
        accountDetailVO.setAuditStatus(result.getJSONObject("data").getInteger("auditStatus"));
        accountDetailVO.setOwnerType(result.getJSONObject("data").getInteger("ownerType"));
        accountDetailVO.setOwnerFullName(result.getJSONObject("data").getString("ownerFullName"));
        accountDetailVO.setIdCard(result.getJSONObject("data").getString("idCard"));
        accountDetailVO.setIdCardExpireDate(result.getJSONObject("data").getString("idCardExpireDate"));
        accountDetailVO.setIdCardFileFrontURL(result.getJSONObject("data").getString("idCardFrontUrl"));
        accountDetailVO.setIdCardFileBackURL(result.getJSONObject("data").getString("idCardBackUrl"));
/*
        AccountDetailVO accountDetailVO = new AccountDetailVO();
        accountDetailVO.setId("12345678");
        accountDetailVO.setAccount("15040196733");
        accountDetailVO.setSourceType("2");
        accountDetailVO.setAuditStatus(1);
        accountDetailVO.setOwnerType(1);
        accountDetailVO.setOwnerFullName("测试123");
        accountDetailVO.setIdCard("210112198707070023");
        accountDetailVO.setIdCardExpireDate("2030-02-05");
        accountDetailVO.setIdCardFileFrontURL("1123");
        accountDetailVO.setIdCardFileBackURL("4456");
*/
        return EResponse.ok(accountDetailVO);
    }

    /**
     * description 帐号实名认证
     * param [RealNameAuthForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/realNameAuth")
    public EResponse realNameAuth(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) RealNameAuthForm realNameAuthForm) {

        if(realNameAuthForm == null) {
            return EResponse.error(ECode.PARAM_ERROR);
        }

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // TODO，Feign调用货主账号密码,目前暂时设定成功
        if("1234qwer".equals(realNameAuthForm.getIdCard())) {
            return EResponse.error(1001, "身份证信息与照片不一致");
        }

        return EResponse.ok();
    }
}