package com.esv.freight.app.module.address.controller;

import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.module.address.form.CreateReceiveAddressForm;
import com.esv.freight.app.module.address.form.DeleteAddressForm;
import com.esv.freight.app.module.address.form.EditReceiveAddressForm;
import com.esv.freight.app.module.address.service.AddressAccountService;
import com.esv.freight.app.module.address.vo.ReceiveAddressListVO;
import com.esv.freight.app.module.address.vo.ReceiveAddressItemVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import java.util.ArrayList;
import java.util.List;

/**
 * @description: 收货地址Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.address.controller.ReceiveAddressController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/13 10:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/address/receive")
@Slf4j
@Validated
public class ReceiveAddressController {

    private AddressAccountService addressAccountService;

    private ReceiveAddressListVO receiveAddressListVO;
    private List<ReceiveAddressItemVO> listReceiveAddressVO;
    private Integer mIndex = 100;

    @Autowired
    public ReceiveAddressController(AddressAccountService addressAccountService) {
        this.addressAccountService = addressAccountService;
        receiveAddressListVO = new ReceiveAddressListVO();
        listReceiveAddressVO = new ArrayList<>();
        receiveAddressListVO.setRecords(listReceiveAddressVO);
    }

    /**
     * description 收货地址列表
     * param [accessToken]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/17 13:00
     **/
    @PostMapping("/list")
    public EResponse list(@RequestHeader("Union-Authorization") String accessToken) {

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1001, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据

        return EResponse.ok(receiveAddressListVO);
    }

    /**
     * description 新增收货地址
     * param [CreateAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/add")
    public EResponse add(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) CreateReceiveAddressForm createAddressForm) {

        if(createAddressForm == null) {
            return EResponse.error(ECode.PARAM_ERROR);
        }

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1001, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据
        for (int i=0; i<listReceiveAddressVO.size(); ++i) {
            ReceiveAddressItemVO vo = listReceiveAddressVO.get(i);
            if(createAddressForm.getAddressName().equals(vo.getAddressName())) {
                return EResponse.error(1002, "重复的地址名称");
            }
        }

        ReceiveAddressItemVO receiveAddressVO = new ReceiveAddressItemVO();
        receiveAddressVO.setAddressName(createAddressForm.getAddressName());
        receiveAddressVO.setId(++mIndex);
        receiveAddressVO.setProvinceCode(createAddressForm.getProvinceCode());
        receiveAddressVO.setCityCode(createAddressForm.getCityCode());
        receiveAddressVO.setDistrictCode(createAddressForm.getDistrictCode());
        receiveAddressVO.setDetailAddress(createAddressForm.getDetailAddress());
        receiveAddressVO.setReceiver(createAddressForm.getReceiver());
        receiveAddressVO.setReceiverPhone(createAddressForm.getReceiverPhone());
        listReceiveAddressVO.add(receiveAddressVO);

        return EResponse.ok();
    }

    /**
     * description 编辑收货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) EditReceiveAddressForm editAddressForm) {

        if(editAddressForm == null) {
            return EResponse.error(ECode.PARAM_ERROR);
        }

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1003, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据
        ReceiveAddressItemVO receiveAddressVO = null;
        for (int i=0; i<listReceiveAddressVO.size(); ++i) {
            ReceiveAddressItemVO addressVO = listReceiveAddressVO.get(i);
            if(addressVO.getId() == editAddressForm.getId()) {
                receiveAddressVO = addressVO;
                break;
            }
        }

        if(receiveAddressVO == null) {
            return EResponse.error(1001, "无效的地址ID");
        }

        receiveAddressVO.setAddressName(editAddressForm.getAddressName());
        receiveAddressVO.setProvinceCode(editAddressForm.getProvinceCode());
        receiveAddressVO.setCityCode(editAddressForm.getCityCode());
        receiveAddressVO.setDistrictCode(editAddressForm.getDistrictCode());
        receiveAddressVO.setDetailAddress(editAddressForm.getDetailAddress());
        receiveAddressVO.setReceiver(editAddressForm.getReceiver());
        receiveAddressVO.setReceiverPhone(editAddressForm.getReceiverPhone());
        return EResponse.ok();
    }

    /**
     * description 删除收货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) DeleteAddressForm deleteAddressForm) {
        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1003, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据
        ReceiveAddressItemVO receiveAddressVO = null;
        int index = 0;
        for (index=0; index<listReceiveAddressVO.size(); ++index) {
            ReceiveAddressItemVO addressVO = listReceiveAddressVO.get(index);
            if(addressVO.getId() == deleteAddressForm.getId()) {
                receiveAddressVO = addressVO;
                break;
            }
        }

        if(receiveAddressVO == null) {
            return EResponse.error(1001, "无效的地址ID");
        }

        listReceiveAddressVO.remove(index);
        return EResponse.ok();
    }

    /**
     * description 默认收货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/default")
    public EResponse defaultAddress(@NotBlank(message="ID不能为空") @RequestParam String id) {
        return EResponse.ok();
    }
}
