package com.esv.freight.app.module.address.controller;

import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.module.address.form.CreateDeliveryAddressForm;
import com.esv.freight.app.module.address.form.DeleteAddressForm;
import com.esv.freight.app.module.address.form.EditDeliveryAddressForm;
import com.esv.freight.app.module.address.service.AddressAccountService;
import com.esv.freight.app.module.address.vo.DeliveryAddressListVO;
import com.esv.freight.app.module.address.vo.DeliveryAddressItemVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import java.util.ArrayList;
import java.util.List;

/**
 * @description: 发货地址Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.address.controller.DeliveryAddressController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/13 10:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/address/delivery")
@Slf4j
@Validated
public class DeliveryAddressController {

    private AddressAccountService addressAccountService;

    private DeliveryAddressListVO deliveryAddressListVO;
    private List<DeliveryAddressItemVO> listDeliveryAddressVO;
    private Integer mIndex = 100;

    @Autowired
    public DeliveryAddressController(AddressAccountService addressAccountService) {
        this.addressAccountService = addressAccountService;
        deliveryAddressListVO = new DeliveryAddressListVO();
        listDeliveryAddressVO = new ArrayList<>();
        deliveryAddressListVO.setRecords(listDeliveryAddressVO);
    }

    /**
     * description 发货地址列表
     * param [accessToken]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/17 13:00
     **/
    @PostMapping("/list")
    public EResponse list(@RequestHeader("Union-Authorization") String accessToken) {

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1001, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据

        return EResponse.ok(deliveryAddressListVO);
    }

    /**
     * description 新增发货地址
     * param [CreateAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/add")
    public EResponse add(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) CreateDeliveryAddressForm createAddressForm) {

        if(createAddressForm == null) {
            return EResponse.error(ECode.PARAM_ERROR);
        }

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1001, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据
        for (int i=0; i<listDeliveryAddressVO.size(); ++i) {
            DeliveryAddressItemVO vo = listDeliveryAddressVO.get(i);
            if(createAddressForm.getAddressName().equals(vo.getAddressName())) {
                return EResponse.error(1002, "重复的地址名称");
            }
        }

        DeliveryAddressItemVO deliveryAddressVO = new DeliveryAddressItemVO();
        deliveryAddressVO.setAddressName(createAddressForm.getAddressName());
        deliveryAddressVO.setId(++mIndex);
        deliveryAddressVO.setProvinceCode(createAddressForm.getProvinceCode());
        deliveryAddressVO.setCityCode(createAddressForm.getCityCode());
        deliveryAddressVO.setDistrictCode(createAddressForm.getDistrictCode());
        deliveryAddressVO.setDetailAddress(createAddressForm.getDetailAddress());
        deliveryAddressVO.setDeliverer(createAddressForm.getDeliverer());
        deliveryAddressVO.setDelivererPhone(createAddressForm.getDelivererPhone());
        listDeliveryAddressVO.add(deliveryAddressVO);

        return EResponse.ok();
    }

    /**
     * description 编辑发货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) EditDeliveryAddressForm editAddressForm) {

        if(editAddressForm == null) {
            return EResponse.error(ECode.PARAM_ERROR);
        }

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1003, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据
        DeliveryAddressItemVO deliveryAddressVO = null;
        for (int i=0; i<listDeliveryAddressVO.size(); ++i) {
            DeliveryAddressItemVO addressVO = listDeliveryAddressVO.get(i);
            if(addressVO.getId() == editAddressForm.getId()) {
                deliveryAddressVO = addressVO;
                break;
            }
        }

        if(deliveryAddressVO == null) {
            return EResponse.error(1001, "无效的地址ID");
        }

        deliveryAddressVO.setAddressName(editAddressForm.getAddressName());
        deliveryAddressVO.setProvinceCode(editAddressForm.getProvinceCode());
        deliveryAddressVO.setCityCode(editAddressForm.getCityCode());
        deliveryAddressVO.setDistrictCode(editAddressForm.getDistrictCode());
        deliveryAddressVO.setDetailAddress(editAddressForm.getDetailAddress());
        deliveryAddressVO.setDeliverer(editAddressForm.getDeliverer());
        deliveryAddressVO.setDelivererPhone(editAddressForm.getDelivererPhone());
        return EResponse.ok();
    }

    /**
     * description 删除发货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) DeleteAddressForm deleteAddressForm) {

        String phone = addressAccountService.getPhoneByAccessToken(accessToken);
        if(phone == null || "".equals(phone)) {
            return EResponse.error(1003, "未找到此用户");
        }

        // TODO，Feign调用货主发货地址列表接口，目前暂时返回假数据
        DeliveryAddressItemVO deliveryAddressVO = null;
        int index = 0;
        for (index=0; index<listDeliveryAddressVO.size(); ++index) {
            DeliveryAddressItemVO addressVO = listDeliveryAddressVO.get(index);
            if(addressVO.getId() == deleteAddressForm.getId()) {
                deliveryAddressVO = addressVO;
                break;
            }
        }

        if(deliveryAddressVO == null) {
            return EResponse.error(1001, "无效的地址ID");
        }

        listDeliveryAddressVO.remove(index);
        return EResponse.ok();
    }

    /**
     * description 默认发货地址
     * param [EditAddressForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 11:00
     **/
    @PostMapping("/default")
    public EResponse defaultAddress(@NotBlank(message="ID不能为空") @RequestParam String id) {
        return EResponse.ok();
    }
}
