package com.esv.freight.app.common.filter;

import com.esv.freight.app.common.wrapper.RestRequestWrapper;
import com.esv.freight.app.common.wrapper.RestResponseWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * @description: Rest请求日志Filter
 * @project: SpringCloudTemplate
 * @name: com.esv.springcloud.template.web.filter.RestLogFilter
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/03/09 17:44
 * @version:1.0
 */
@Slf4j
public class RestLogFilter implements Filter {

    @Override
    public void init(FilterConfig filterConfig) {

    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        RestRequestWrapper requestWrapper = new RestRequestWrapper((HttpServletRequest)servletRequest);
        RestResponseWrapper responseWrapper = new RestResponseWrapper((HttpServletResponse) servletResponse);

        String reqContentType = StringUtils.trimToEmpty(requestWrapper.getContentType()).toLowerCase();
        if (reqContentType.contains("multipart/form-data")) {
            log.info("multipart/form-data request");
        } else {
            // 日志输出请求
            logReq(requestWrapper);
        }

        filterChain.doFilter(requestWrapper, responseWrapper);

        String resContentType = StringUtils.trimToEmpty(responseWrapper.getContentType());
        if (resContentType.contains("text") || resContentType.contains("xml")
                || resContentType.contains("json")) {
            // 日志输出返回
            try {
                logRes(requestWrapper, responseWrapper);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        } else {
            log.info("Response ContentType: {}", resContentType);
        }
    }

    @Override
    public void destroy() {

    }

    private String getPostBodyStr(HttpServletRequest request) {
        String bodyStr = null;
        StringBuilder sb = new StringBuilder();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
            inputStream = request.getInputStream();
            reader = new BufferedReader(new InputStreamReader(inputStream, Charset.forName("UTF-8")));
            String line = "";
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }

            if (0 == sb.length()) {
                Map<String, String> bodyMap = new HashMap<>();
                Map<String, String[]> parameterMap = request.getParameterMap();
                for (Map.Entry<String, String[]> entry : parameterMap.entrySet()) {
                    for (String value : entry.getValue()) {
                        bodyMap.put(entry.getKey(), value);
                    }
                }
                bodyStr = bodyMap.toString();
            } else {
                bodyStr = sb.toString();
            }
        } catch (IOException e) {
            log.error("解析post参数时发生错误：{}", e.getMessage());
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return bodyStr;
    }

    private void logReq(RestRequestWrapper requestWrapper) {
        String url = requestWrapper.getRequestURI();
        String method = requestWrapper.getMethod();
        if ("GET".equalsIgnoreCase(method)) {
            Enumeration em = requestWrapper.getParameterNames();
            String reqParams = "";
            while (em.hasMoreElements()) {
                String k = em.nextElement().toString();
                String v = requestWrapper.getParameter(k);
                reqParams += "&" + k + "=" + v;
            }
            reqParams = reqParams.replaceFirst("&", "");
            if (url.startsWith("/actuator")) {
                log.debug("[IP={}]收到{}请求，url：{}，params：{}", getHttpClientIp(requestWrapper), method, url, reqParams);
            } else {
                log.info("[IP={}]收到{}请求，url：{}，params：{}", getHttpClientIp(requestWrapper), method, url, reqParams);
            }
        } else if ("POST".equalsIgnoreCase(method)) {
            log.info("[IP={}]收到{}请求，url：{}，body：{}", getHttpClientIp(requestWrapper), method, url, getPostBodyStr(requestWrapper));
        } else {

        }
    }

    private void logRes(RestRequestWrapper requestWrapper, RestResponseWrapper responseWrapper) throws Exception {
        byte[] bytes = responseWrapper.getBody();
        String resStr = new String(bytes,"utf-8");
        String url = requestWrapper.getRequestURI();
        if (url.startsWith("/actuator")) {
            log.debug("请求响应：{}", resStr);
        } else {
            log.info("请求响应：{}", resStr);
        }
    }

    private String getHttpClientIp(HttpServletRequest req){
        String ip = req.getHeader("x-forwarded-for");
        if(ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = req.getHeader("Proxy-Client-IP");
        }
        if(ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = req.getHeader("WL-Proxy-Client-IP");
        }
        if(ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = req.getRemoteAddr();
        }
        return ip;
    }
}