package com.esv.freight.app.module.account.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.feign.DriverInterface;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.CustomToken;
import com.esv.freight.app.module.account.form.DriverAuthForm;
import com.esv.freight.app.module.account.form.LoginForm;
import com.esv.freight.app.module.account.form.OwnerAuthForm;
import com.esv.freight.app.module.account.form.RefreshTokenForm;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginByPwd;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginBySms;
import com.esv.freight.app.module.account.vo.DriverAccountDetailVO;
import com.esv.freight.app.module.account.vo.LoginVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * @description: 账号Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.account.controller.DriverAccountController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/14 14:00
 * @version:1.0
 */
@RestController
@RequestMapping("/driverBackend/account")
@Slf4j
@Validated
public class DriverAccountController {

    private DriverInterface driverInterface;
    private NoticeInterface noticeInterface;
    private AppLoginService appLoginService;

    @Autowired
    public DriverAccountController(DriverInterface driverInterface, NoticeInterface noticeInterface, AppLoginService appLoginService) {
        this.noticeInterface = noticeInterface;
        this.appLoginService = appLoginService;
        this.appLoginService = appLoginService;
    }

    /**
     * description 短信验证码登录
     * param [LoginBySmsForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/login/loginBySms")
    public EResponse loginBySms(@RequestHeader @RequestBody(required=false) @Validated(ValidatorLoginBySms.class) LoginForm loginForm) {

        // 调用通知服务验证短信接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", loginForm.getPhone());
        reqJson.put("type", "login");
        reqJson.put("captcha", loginForm.getSmsCode());
        log.info(reqJson.toJSONString());
        JSONObject resultCheck = noticeInterface.checkSmsCaptcha(reqJson);
        log.info(resultCheck.toJSONString());

        if(resultCheck.getInteger("code") != 200) {
            return EResponse.error(resultCheck.getInteger("code"), resultCheck.getString("message"));
        }

        // 调用注册帐号接口
        reqJson.clear();
        reqJson.put("account", loginForm.getPhone());
        JSONObject resultRegister = driverInterface.accountRegister(reqJson);

        // 1001表示 帐号已存在
        if(resultRegister.getInteger("code") != 200 && resultRegister.getInteger("code") != 1001) {
            return EResponse.error(resultRegister.getInteger("code"), resultRegister.getString("message"));
        }

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号密码登录RequestHeader
     * param [LoginByPwdForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/login/loginByPwd")
    public EResponse loginByPwd(@RequestBody(required=false) @Validated(ValidatorLoginByPwd.class) LoginForm loginForm) {

        // 调用帐号密码校验接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("account", loginForm.getPhone());
        reqJson.put("password", loginForm.getPwd());
        JSONObject result = driverInterface.checkAccountPwd(reqJson);

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号登出
     * param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/logout")
    public EResponse logout() {
        CustomToken customToken = ReqUtils.getTokenInfo();
        appLoginService.logout(customToken.getAccessToken());
        return EResponse.ok();
    }

    /**
     * description 刷新Token
     * param [refreshToken] 刷新Token
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 09:00
     **/
    @PostMapping("/token/refresh")
    public EResponse refresh(@RequestBody(required=false) @Validated(ValidatorInsert.class) RefreshTokenForm refreshTokenForm) {

        CustomToken customToken = ReqUtils.getTokenInfo();
        LoginVO loginByPwdVO = appLoginService.refreshToken(customToken.getAccessToken(), refreshTokenForm);
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 获取帐号详情
     * param []
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 15:00
     **/
    @PostMapping("/detail")
    public EResponse detail() {
        String accessToken = ReqUtils.getTokenInfo().getAccessToken();
        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        String phone = ReqUtils.getTokenInfo().getAccount();

        // 调用帐号密码校验接口
        JSONObject reqJsonDetail = new JSONObject();
        reqJsonDetail.put("account", phone);
        log.info(reqJsonDetail.toJSONString());
        JSONObject result = driverInterface.getAccountDetail(reqJsonDetail);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        DriverAccountDetailVO detailVO = new DriverAccountDetailVO();
        detailVO.setId(String.valueOf(result.getJSONObject("data").getLong("id")));
        detailVO.setCarrierId(result.getJSONObject("data").getLong("carrierId"));
        detailVO.setAccount(result.getJSONObject("data").getString("account"));
        detailVO.setSourceType(result.getJSONObject("data").getString("sourceType"));
        detailVO.setAuditStatus(result.getJSONObject("data").getInteger("auditStatus"));
        detailVO.setAuthenticateStatus(result.getJSONObject("data").getInteger("authenticateStatus"));
        detailVO.setName(result.getJSONObject("data").getString("name"));
        detailVO.setIdCard(result.getJSONObject("data").getString("idCard"));
        detailVO.setIdCardExpireDate(result.getJSONObject("data").getString("idCardExpireDate"));
        detailVO.setIdCardFrontUrl(result.getJSONObject("data").getString("idCardFrontUrl"));
        detailVO.setIdCardBackUrl(result.getJSONObject("data").getString("idCardBackUrl"));
        detailVO.setSettlementType(result.getJSONObject("data").getInteger("settlementType"));
        detailVO.setSex(result.getJSONObject("data").getInteger("sex"));
        detailVO.setBirthDate(result.getJSONObject("data").getString("birthDate"));
        detailVO.setNation(result.getJSONObject("data").getString("nation"));
        detailVO.setNativePlace(result.getJSONObject("data").getString("nativePlace"));
        detailVO.setProvinceCode(result.getJSONObject("data").getString("provinceCode"));
        detailVO.setCityCode(result.getJSONObject("data").getString("cityCode"));
        detailVO.setDistrictCode(result.getJSONObject("data").getString("districtCode"));
        detailVO.setDetailAddress(result.getJSONObject("data").getString("detailAddress"));
        detailVO.setDrivingLicense(result.getJSONObject("data").getString("drivingLicense"));
        detailVO.setDrivingLicenseType(result.getJSONObject("data").getInteger("drivingLicenseType"));
        detailVO.setDrivingLicenseStartDate(result.getJSONObject("data").getString("drivingLicenseStartDate"));
        detailVO.setDrivingLicenseEndDate(result.getJSONObject("data").getString("drivingLicenseEndDate"));
        detailVO.setDrivingLicenseIssueDepartment(result.getJSONObject("data").getString("drivingLicenseIssueDepartment"));
        detailVO.setDrivingLicenseInitDate(result.getJSONObject("data").getString("drivingLicenseInitDate"));
        detailVO.setDrivingLicenseUrl(result.getJSONObject("data").getString("drivingLicenseUrl"));
        detailVO.setCertificateVehicle(result.getJSONObject("data").getInteger("certificateVehicle"));
        detailVO.setCertificateEndDate(result.getJSONObject("data").getString("certificateEndDate"));
        detailVO.setCertificateNumber(result.getJSONObject("data").getString("certificateNumber"));
        detailVO.setCertificateUrl(result.getJSONObject("data").getString("certificateUrl"));
        return EResponse.ok(detailVO);
    }

    /**
     * description 帐号实名认证
     * param [RealNameAuthForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/realNameAuth")
    public EResponse realNameAuth(@RequestBody(required=false) @Validated(ValidatorInsert.class) DriverAuthForm driverAuthForm) {

        String accessToken = ReqUtils.getTokenInfo().getAccessToken();
        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用编辑帐号信息接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", driverAuthForm.getId());

        log.info(reqJson.toJSONString());
        JSONObject result = driverInterface.updateAccountInfo(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }
}
