package com.esv.freight.app.module.account.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.component.TokenComponent;
import com.esv.freight.app.common.constants.AccountConstants;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.FeignUtils;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.common.validator.groups.ValidatorUpdate;
import com.esv.freight.app.feign.CarrierInterface;
import com.esv.freight.app.feign.DriverInterface;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.form.DriverAuthForm;
import com.esv.freight.app.module.account.form.LoginForm;
import com.esv.freight.app.module.account.form.RefreshTokenForm;
import com.esv.freight.app.module.account.pojo.DriverAccountInfoPojo;
import com.esv.freight.app.module.account.pojo.TokenInfoPojo;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.validator.groups.ValidatorAccountExist;
import com.esv.freight.app.module.account.validator.groups.ValidatorDriverLoginBySms;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginByPwd;
import com.esv.freight.app.module.account.vo.AccountExistVO;
import com.esv.freight.app.module.account.vo.CarrierInfoBriefVO;
import com.esv.freight.app.module.account.vo.DriverAccountDetailVO;
import com.esv.freight.app.module.account.vo.LoginVO;
import com.esv.freight.app.module.message.form.AccountDeviceForm;
import com.esv.freight.app.module.message.service.AccountDeviceInfoService;
import com.esv.freight.app.module.message.service.BusinessMessageService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 账号Controller
 * @project: freight-app-service
 * @name: com.esv.freight.module.ownerBackend.account.controller.DriverAccountController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/14 14:00
 * @version:1.0
 */
@RestController
@RequestMapping("/driverBackend/account")
@Slf4j
@Validated
public class DriverAccountController {

    private DriverInterface driverInterface;
    private NoticeInterface noticeInterface;
    private AppLoginService appLoginService;
    private CarrierInterface carrierInterface;
    private TokenComponent tokenComponent;
    private AccountDeviceInfoService accountDeviceInfoService;
    private BusinessMessageService businessMessageService;

    @Autowired
    public DriverAccountController(TokenComponent tokenComponent, AccountDeviceInfoService accountDeviceInfoService, CarrierInterface carrierInterface, DriverInterface driverInterface, NoticeInterface noticeInterface, AppLoginService appLoginService) {
        this.noticeInterface = noticeInterface;
        this.appLoginService = appLoginService;
        this.driverInterface = driverInterface;
        this.carrierInterface = carrierInterface;
        this.tokenComponent = tokenComponent;
        this.accountDeviceInfoService = accountDeviceInfoService;
        this.businessMessageService = businessMessageService;
    }

    /**
     * description 短信验证码登录
     * param [LoginForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/login/loginBySms")
    public EResponse loginBySms(@RequestHeader @RequestBody(required=false) @Validated(ValidatorDriverLoginBySms.class) LoginForm loginForm) {

        // 1:调用通知服务验证短信接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", loginForm.getPhone());
        reqJson.put("type", "login");
        reqJson.put("captcha", loginForm.getSmsCode());
        JSONObject result;
        try {
            result = noticeInterface.checkSmsCaptcha(reqJson);
        } catch (Exception e) {
            log.error("Feign请求时发生错误：" + e.getMessage(), e);
            return EResponse.error();
        }

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        // 2:查询司机详情(通过帐号)
        reqJson.clear();
        reqJson.put("account", loginForm.getPhone());
        try {
            result = driverInterface.getDetailByAccount(reqJson);
        } catch (Exception e) {
            log.error("Feign请求时发生错误：" + e.getMessage(), e);
            return EResponse.error();
        }

        // 3:表示账号未注册
        if(result.getInteger("code") == 1001) {
            // 调用注册帐号接口
            reqJson.clear();
            reqJson.put("account", loginForm.getPhone());
            reqJson.put("carrierId", loginForm.getCarrierId());
            try {
                result = driverInterface.register(reqJson);
            } catch (Exception e) {
                log.error("Feign请求时发生错误：" + e.getMessage(), e);
                return EResponse.error();
            }

            if(!FeignUtils.isFeignSuccess(result)) {
                return FeignUtils.getFeignEResponse(result);
            }

            DriverAccountInfoPojo pojo = new DriverAccountInfoPojo();
            pojo.setId(result.getJSONObject("data").getLong("id"));
            pojo.setCarrierId(loginForm.getCarrierId());
            pojo.setAccount(loginForm.getPhone());
            pojo.setLoginMode(AccountConstants.ACCOUNT_LOGIN_MODE_SMS);
            LoginVO vo = appLoginService.driverLogin(pojo);

            // 提交推送标识
            AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
            accountDeviceForm.setAccount(loginForm.getPhone());
            accountDeviceForm.setAccountId(pojo.getId());
            accountDeviceForm.setDeviceSn(loginForm.getPushId());
            accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

            return EResponse.ok(vo);
        }

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        DriverAccountInfoPojo driverAccountInfoPojo = JSONObject.toJavaObject(FeignUtils.getFeignDataJson(result), DriverAccountInfoPojo.class);
        if (AccountConstants.AUDIT_STATUS_CHECK_SUCCESS.equals(driverAccountInfoPojo.getAuditStatus())) {
            // 校验帐号状态：1-正常、2-停用
            if (AccountConstants.ACCOUNT_STATUS_BLOCK.equals(driverAccountInfoPojo.getAccountStatus())) {
                return EResponse.error(1004, "帐号已停用");
            }
        }

        // 登录
        driverAccountInfoPojo.setLoginMode(AccountConstants.ACCOUNT_LOGIN_MODE_SMS);
        LoginVO vo = appLoginService.driverLogin(driverAccountInfoPojo);

        // 提交推送标识
        AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
        accountDeviceForm.setAccount(loginForm.getPhone());
        accountDeviceForm.setAccountId(driverAccountInfoPojo.getId());
        accountDeviceForm.setDeviceSn(loginForm.getPushId());
        accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

        return EResponse.ok(vo);
    }

    /**
     * description 帐号密码登录
     * param [LoginForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/login/loginByPwd")
    public EResponse loginByPwd(@RequestBody(required=false) @Validated(ValidatorLoginByPwd.class) LoginForm loginForm) {

        // 1:调用帐号密码校验接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("account", loginForm.getPhone());
        reqJson.put("password", loginForm.getPwd());
        JSONObject result;
        try {
            result = driverInterface.accountCheck(reqJson);
        } catch (Exception e) {
            log.error("Feign请求时发生错误：" + e.getMessage(), e);
            return EResponse.error();
        }

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        DriverAccountInfoPojo driverAccountInfoPojo = JSONObject.toJavaObject(FeignUtils.getFeignDataJson(result), DriverAccountInfoPojo.class);
        if (AccountConstants.AUDIT_STATUS_CHECK_SUCCESS.equals(driverAccountInfoPojo.getAuditStatus())) {
            // 校验帐号状态：1-正常、2-停用
            if (AccountConstants.ACCOUNT_STATUS_BLOCK.equals(driverAccountInfoPojo.getAccountStatus())) {
                return EResponse.error(1003, "帐号已停用");
            }
        }

        // 登录
        driverAccountInfoPojo.setLoginMode(AccountConstants.ACCOUNT_LOGIN_MODE_PWD);
        LoginVO vo = appLoginService.driverLogin(driverAccountInfoPojo);

        // 提交推送标识
        AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
        accountDeviceForm.setAccount(loginForm.getPhone());
        accountDeviceForm.setAccountId(driverAccountInfoPojo.getId());
        accountDeviceForm.setDeviceSn(loginForm.getPushId());
        accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

        return EResponse.ok(vo);
    }

    /**
     * description 检验手机号是否已存在，司机短信验证码登录时调用，如果已存在直接注册，如果不存在需要APP先选择一个承运商
     * param [LoginForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/06 10:00
     **/
    @PostMapping("/login/check/phone")
    public EResponse accountIsExist(@RequestBody(required=false) @Validated(ValidatorAccountExist.class) LoginForm loginForm) {

        // 调用查询司机详情-通过帐号接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("account", loginForm.getPhone());
        JSONObject result = driverInterface.getDetailByAccount(reqJson);

        // 1001-表示账号不存在
        if(result.getInteger("code") == 1001) {
            AccountExistVO vo = new AccountExistVO();
            vo.setAccountStatus(2);
            return EResponse.ok(vo);
        }

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        AccountExistVO vo = new AccountExistVO();
        vo.setAccountStatus(result.getJSONObject("data").getInteger("accountStatus"));
        vo.setId(result.getJSONObject("data").getLong("id"));
        vo.setCarrierId(result.getJSONObject("data").getLong("carrierId"));
        vo.setCarrierName(result.getJSONObject("data").getString("carrierName"));
        return EResponse.ok(vo);
    }

    /**
     * description 帐号登出
     * param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 14:00
     **/
    @PostMapping("/logout")
    public EResponse logout() {
        accountDeviceInfoService.accountLogout();
        appLoginService.logout();
        return EResponse.ok();
    }

    /**
     * description 刷新Token
     * param [refreshToken] 刷新Token
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 09:00
     **/
    @PostMapping("/token/refresh")
    public EResponse refresh(@RequestBody(required=false) @Validated(ValidatorUpdate.class) RefreshTokenForm refreshTokenForm) {

        LoginVO loginByPwdVO = appLoginService.refreshToken(refreshTokenForm);

        TokenInfoPojo tokenInfoPojo = tokenComponent.getTokenInfo();
        // 提交推送标识
        AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
        accountDeviceForm.setAccount(tokenInfoPojo.getAccount());
        accountDeviceForm.setAccountId(tokenInfoPojo.getAccountId());
        accountDeviceForm.setDeviceSn(refreshTokenForm.getPushId());
        accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 获取帐号详情
     * param []
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/14 15:00
     **/
    @PostMapping("/detail")
    public EResponse detail() {

        TokenInfoPojo tokenInfoPojo = tokenComponent.getTokenInfo();
        String phone = tokenInfoPojo.getAccount();

        // 调用获取账号详情接口
        JSONObject reqJsonDetail = new JSONObject();
        reqJsonDetail.put("account", phone);
        log.info(reqJsonDetail.toJSONString());
        JSONObject result = driverInterface.getDetailByAccount(reqJsonDetail);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        DriverAccountDetailVO detailVO = new DriverAccountDetailVO();
        detailVO.setId(String.valueOf(result.getJSONObject("data").getLong("id")));
        detailVO.setCarrierId(result.getJSONObject("data").getLong("carrierId"));
        detailVO.setCarrierName(result.getJSONObject("data").getString("carrierName"));
        detailVO.setAccount(result.getJSONObject("data").getString("account"));
        detailVO.setAccountStatus(result.getJSONObject("data").getInteger("accountStatus"));
        detailVO.setSourceType(result.getJSONObject("data").getInteger("sourceType"));
        detailVO.setAuditStatus(result.getJSONObject("data").getInteger("auditStatus"));
        detailVO.setName(result.getJSONObject("data").getString("name"));
        detailVO.setIdCard(result.getJSONObject("data").getString("idCard"));
        detailVO.setIdCardExpireDate(result.getJSONObject("data").getString("idCardExpireDate"));
        detailVO.setIdCardFrontUrl(result.getJSONObject("data").getString("idCardFrontUrl"));
        detailVO.setIdCardBackUrl(result.getJSONObject("data").getString("idCardBackUrl"));
        detailVO.setSettlementType(result.getJSONObject("data").getInteger("settlementType"));
        detailVO.setSex(result.getJSONObject("data").getInteger("sex"));
        detailVO.setBirthDate(result.getJSONObject("data").getString("birthDate"));
        detailVO.setNation(result.getJSONObject("data").getString("nation"));
        detailVO.setNativePlace(result.getJSONObject("data").getString("nativePlace"));
        detailVO.setProvinceCode(result.getJSONObject("data").getString("provinceCode"));
        detailVO.setCityCode(result.getJSONObject("data").getString("cityCode"));
        detailVO.setDistrictCode(result.getJSONObject("data").getString("districtCode"));
        detailVO.setDetailAddress(result.getJSONObject("data").getString("detailAddress"));
        detailVO.setDrivingLicense(result.getJSONObject("data").getString("drivingLicense"));
        detailVO.setDrivingLicenseType(result.getJSONObject("data").getInteger("drivingLicenseType"));
        detailVO.setDrivingLicenseStartDate(result.getJSONObject("data").getString("drivingLicenseStartDate"));
        detailVO.setDrivingLicenseEndDate(result.getJSONObject("data").getString("drivingLicenseEndDate"));
        detailVO.setDrivingLicenseIssueDepartment(result.getJSONObject("data").getString("drivingLicenseIssueDepartment"));
        detailVO.setDrivingLicenseInitDate(result.getJSONObject("data").getString("drivingLicenseInitDate"));
        detailVO.setDrivingLicenseUrl(result.getJSONObject("data").getString("drivingLicenseUrl"));
        detailVO.setCertificateVehicle(result.getJSONObject("data").getInteger("certificateVehicle"));
        detailVO.setCertificateEndDate(result.getJSONObject("data").getString("certificateEndDate"));
        detailVO.setCertificateNumber(result.getJSONObject("data").getString("certificateNumber"));
        detailVO.setCertificateUrl(result.getJSONObject("data").getString("certificateUrl"));
        detailVO.setHasUnReadMessage(businessMessageService.hasUnReadMessage(Long.valueOf(detailVO.getId())));
        return EResponse.ok(detailVO);
    }

    /**
     * description 帐号实名认证
     * param [RealNameAuthForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/realNameAuth")
    public EResponse realNameAuth(@RequestBody(required=false) @Validated(ValidatorUpdate.class) DriverAuthForm driverAuthForm) {

        // 调用编辑帐号信息接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", driverAuthForm.getId());
        reqJson.put("name", driverAuthForm.getName());
        reqJson.put("idCard", driverAuthForm.getIdCard());
        reqJson.put("idCardExpireDate", driverAuthForm.getIdCardExpireDate());
        reqJson.put("provinceCode", driverAuthForm.getProvinceCode());
        reqJson.put("cityCode", driverAuthForm.getCityCode());
        reqJson.put("districtCode", driverAuthForm.getDistrictCode());
        reqJson.put("detailAddress", driverAuthForm.getDetailAddress());
        reqJson.put("settlementType", driverAuthForm.getSettlementType());
        reqJson.put("sex", driverAuthForm.getSex());
        reqJson.put("birthDate", driverAuthForm.getBirthDate());
        reqJson.put("drivingLicense", driverAuthForm.getDrivingLicense());
        reqJson.put("drivingLicenseType", driverAuthForm.getDrivingLicenseType());
        reqJson.put("drivingLicenseStartDate", driverAuthForm.getDrivingLicenseStartDate());
        reqJson.put("drivingLicenseEndDate", driverAuthForm.getDrivingLicenseEndDate());
        reqJson.put("drivingLicenseIssueDepartment", driverAuthForm.getDrivingLicenseIssueDepartment());
        reqJson.put("certificateVehicle", driverAuthForm.getCertificateVehicle());
        reqJson.put("certificateEndDate", driverAuthForm.getCertificateEndDate());
        reqJson.put("certificateNumber", driverAuthForm.getCertificateNumber());
        reqJson.put("certificateUrl", driverAuthForm.getCertificateUrl());
        reqJson.put("drivingLicenseUrl", driverAuthForm.getDrivingLicenseUrl());
        reqJson.put("idCardFrontUrl", driverAuthForm.getIdCardFrontUrl());
        reqJson.put("idCardBackUrl", driverAuthForm.getIdCardBackUrl());

        JSONObject result = driverInterface.edit(reqJson);

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 获取承运商列表
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/carrier/list")
    public EResponse getCarrierList() {

        // 调用获取承运商列表接口
        JSONObject reqJson = new JSONObject();
        JSONObject result = carrierInterface.all(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        List<CarrierInfoBriefVO> carrierInfoBriefVOList = new ArrayList<>();
        JSONArray items = result.getJSONArray("data");
        if(items.size() > 0) {
            for(int i=0; i<items.size(); ++i) {
                CarrierInfoBriefVO vo = new CarrierInfoBriefVO();
                JSONObject object = items.getJSONObject(i);
                vo.setId(object.getLong("id"));
                vo.setAccount(object.getString("account"));
                vo.setAccountStatus(object.getInteger("accountStatus"));
                vo.setCarrierNumber(object.getString("carrierNumber"));
                vo.setCarrierName(object.getString("carrierName"));
                carrierInfoBriefVOList.add(vo);
            }
        }
        return EResponse.ok(carrierInfoBriefVOList);
    }

    /**
     * description 司机账号停用
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/06 11:00
     **/
    @PostMapping("/stop")
    public EResponse stopUsingAccount(@RequestBody(required=false) @Validated(ValidatorAccountExist.class) LoginForm loginForm) {

        appLoginService.stopUsing(loginForm.getPhone(), AccountConstants.ACCOUNT_TYPE_DRIVER);
        return EResponse.ok();
    }
}
