package com.esv.freight.app.common.component;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.esv.freight.app.common.constants.AccountConstants;
import com.esv.freight.app.common.exception.EException;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.util.AESSecretUtils;
import com.esv.freight.app.common.util.AesUtils;
import com.esv.freight.app.common.util.DateUtils;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.module.account.entity.AppLoginEntity;
import com.esv.freight.app.module.account.pojo.TokenInfoPojo;
import com.esv.freight.app.module.account.service.AppLoginService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.Date;

/**
 * @description: Token组件
 * @project: app-service
 * @name: com.esv.freight.app.common.component.TokenComponent
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/05/14 11:26
 * @version:1.0
 */
@Component
@Slf4j
public class TokenComponent {

    @Value("${spring.application.name}")
    private String applicationName;

    // 为了和终端加密/解密统一，密码长度控制为16位
    @Value("${aes.sha1prng.key:ADGJLPIYRE24680A}")
    private String AES_KEY;

    /**
     * 访问Token：1天
     **/
    @Value("${token.access.valid.time:1}")
    private Integer accessTokenValidTime;

    /**
     * 刷新Token：30天
     **/
    @Value("${token.refresh.valid.time:30}")
    private Integer refreshTokenValidTime;

    private static final Integer TOKEN_TIME = 86400;

    @Autowired
    private RedisComponent redisComponent;

    @Autowired
    private AppLoginService appLoginService;

    /**
     * description 创建Token信息
     * param [tokenInfoPojo, driverAccountInfoPojo]
     * return void
     * author Administrator
     * createTime 2020/05/14 13:48
     **/
    public void generateTokenInfo(TokenInfoPojo tokenInfoPojo) {
        // 构建Token信息
        tokenInfoPojo.setIp(ReqUtils.getHttpClientIp());
        tokenInfoPojo.setSn(ReqUtils.getRequestHeader("mobileId"));
        tokenInfoPojo.setDeviceType(Integer.parseInt(ReqUtils.getRequestHeader("Source-Type")));
        tokenInfoPojo.setOsType(ReqUtils.getRequestHeader("osType"));
        tokenInfoPojo.setOsVersion(ReqUtils.getRequestHeader("osVer"));
        tokenInfoPojo.setAppVersion(ReqUtils.getRequestHeader("cVer"));
        tokenInfoPojo.setAccountType(Integer.parseInt(ReqUtils.getRequestHeader("accountType")));
        String accessToken = AesUtils.aesEncrypt(this.getAccessTokenOriginalContent(tokenInfoPojo), AES_KEY);
        tokenInfoPojo.setAccessToken(accessToken);
        String refreshToken = AesUtils.aesEncrypt(this.getRefreshTokenOriginalContent(tokenInfoPojo), AES_KEY);
        tokenInfoPojo.setRefreshToken(refreshToken);
        Date loginTime = new Date();
        tokenInfoPojo.setLoginTime(loginTime.getTime());
        Date accessTokenValidTime = DateUtils.plusDays(loginTime, this.accessTokenValidTime);
        Date refreshTokenValidTime = DateUtils.plusDays(loginTime, this.refreshTokenValidTime);
        tokenInfoPojo.setAccessTokenValidTime(accessTokenValidTime.getTime());
        tokenInfoPojo.setRefreshTokenValidTime(refreshTokenValidTime.getTime());

        // 缓存Token信息
        String cacheKey = this.getTokenInfoCacheKey(tokenInfoPojo.getAccount(), tokenInfoPojo.getAccountType());
        redisComponent.set(cacheKey, tokenInfoPojo.toString(), this.accessTokenValidTime * TOKEN_TIME);
    }

    /**
     * description 刷新Token信息
     * param [tokenInfoPojo, driverAccountInfoPojo]
     * return void
     * author Administrator
     * createTime 2020/06/04 09:52
     **/
    public void refreshTokenInfo(TokenInfoPojo tokenInfoPojo) {
        // 构建Token信息
        tokenInfoPojo.setIp(ReqUtils.getHttpClientIp());
        tokenInfoPojo.setSn(ReqUtils.getRequestHeader("mobileId"));
        tokenInfoPojo.setDeviceType(Integer.parseInt(ReqUtils.getRequestHeader("Source-Type")));
        tokenInfoPojo.setOsType(ReqUtils.getRequestHeader("osType"));
        tokenInfoPojo.setOsVersion(ReqUtils.getRequestHeader("osVer"));
        tokenInfoPojo.setAppVersion(ReqUtils.getRequestHeader("cVer"));
        tokenInfoPojo.setAccountType(Integer.parseInt(ReqUtils.getRequestHeader("accountType")));
        String accessToken = AesUtils.aesEncrypt(this.getAccessTokenOriginalContent(tokenInfoPojo), AES_KEY);
        tokenInfoPojo.setAccessToken(accessToken);
        String refreshToken = AesUtils.aesEncrypt(this.getRefreshTokenOriginalContent(tokenInfoPojo), AES_KEY);
        tokenInfoPojo.setRefreshToken(refreshToken);
        Date refreshTime = new Date();
        tokenInfoPojo.setRefreshTime(refreshTime.getTime());
        Date accessTokenValidTime = DateUtils.plusDays(refreshTime, this.accessTokenValidTime);
        Date refreshTokenValidTime = DateUtils.plusDays(refreshTime, this.refreshTokenValidTime);
        tokenInfoPojo.setAccessTokenValidTime(accessTokenValidTime.getTime());
        tokenInfoPojo.setRefreshTokenValidTime(refreshTokenValidTime.getTime());

        // 缓存Token信息
        String cacheKey = this.getTokenInfoCacheKey(tokenInfoPojo.getAccount(), tokenInfoPojo.getAccountType());
        redisComponent.set(cacheKey, tokenInfoPojo.toString(), this.accessTokenValidTime * TOKEN_TIME);
    }

    /**
     * description 校验Token是否过期
     * param [accessToken]
     * return void
     * author Administrator
     * createTime 2020/05/14 17:37
     **/
    public void checkAccessToken(String accessToken) throws EException {
        // 获取Token信息
        TokenInfoPojo tokenInfoPojo = this.getTokenInfo(accessToken);
        if (null == tokenInfoPojo) {
            throw new EException(ECode.TOKEN_INVALID.code(), ECode.TOKEN_INVALID.message());
        }

        // 校验Token是否过期
        Long accessTokenValidTime = tokenInfoPojo.getAccessTokenValidTime();
        if (System.currentTimeMillis() > accessTokenValidTime) {
            throw new EException(ECode.TOKEN_EXPIRED.code(), ECode.TOKEN_EXPIRED.message());
        }
    }

    public TokenInfoPojo getTokenInfo() {
        String token = ReqUtils.getRequestHeader("Union-Authorization");
        if(StringUtils.isEmpty(token)) {
            return null;
        }
        else {
            return this.getTokenInfo(token);
        }
    }

    /**
     * description 获取Token信息
     * param [accessToken]
     * return com.esv.freight.app.module.account.pojo.TokenInfoPojo
     * author Administrator
     * createTime 2020/05/14 17:41
     **/
    public TokenInfoPojo getTokenInfo(String accessToken) {
        if (StringUtils.isBlank(accessToken)) {
            return null;
        }
        // 解密Token并基础校验
        String decryptToken = AesUtils.aesDecrypt(accessToken, AES_KEY);
        if (StringUtils.isBlank(decryptToken)) {
            return null;
        }
        String[] tokenArr = decryptToken.split(",");
        if (4 != tokenArr.length || 11 != tokenArr[0].length()) {
            return null;
        }
        if (!AccountConstants.ACCOUNT_TYPE_DRIVER.equals(Integer.parseInt(tokenArr[1]))
                && !AccountConstants.ACCOUNT_TYPE_GOODS_OWNER.equals(Integer.parseInt(tokenArr[1]))) {
            return null;
        }
        String account = tokenArr[0];
        Integer accountType = Integer.parseInt(tokenArr[1]);
        return this.getTokenInfo(account, accountType);
    }

    /**
     * description 获取Token信息
     * param [account, accountType]
     * return com.esv.freight.app.module.account.pojo.TokenInfoPojo
     * author Administrator
     * createTime 2020/05/14 17:34
     **/
    public TokenInfoPojo getTokenInfo(String account, Integer accountType) {
        String cacheKey = this.getTokenInfoCacheKey(account, accountType);
        String cacheInfo = (String) redisComponent.get(cacheKey);
        TokenInfoPojo tokenInfoPojo = null;
        if (null == cacheInfo) {
            AppLoginEntity appLoginEntity = appLoginService.getBaseMapper().selectOne(new QueryWrapper<AppLoginEntity>().lambda()
                    .eq(AppLoginEntity::getAccountType, accountType)
                    .eq(AppLoginEntity::getPhone, account));
            if (null != appLoginEntity) {
                tokenInfoPojo = new TokenInfoPojo();
                BeanUtils.copyProperties(appLoginEntity, tokenInfoPojo);
                try {
                    tokenInfoPojo.setAccountId(appLoginEntity.getId());
                    tokenInfoPojo.setAccount(appLoginEntity.getPhone());
                    tokenInfoPojo.setLoginTime(appLoginEntity.getLoginTime().getTime());
                    tokenInfoPojo.setRefreshTime(appLoginEntity.getRefreshTime().getTime());
                    tokenInfoPojo.setAccessTokenValidTime(appLoginEntity.getAccessTokenValidTime().getTime());
                    tokenInfoPojo.setRefreshTokenValidTime(appLoginEntity.getRefreshTokenValidTime().getTime());

                    // 缓存Token信息
                    redisComponent.set(cacheKey, tokenInfoPojo.toString(), this.accessTokenValidTime * TOKEN_TIME);
                }
                catch (Exception e) {
                    return null;
                }
            }
        } else {
            tokenInfoPojo = JSONObject.toJavaObject(JSONObject.parseObject(cacheInfo), TokenInfoPojo.class);
        }
        return tokenInfoPojo;
    }

    /**
     * description 获取访问Token原始内容
     * param [tokenInfoPojo]
     * return java.lang.String
     * author Administrator
     * createTime 2020/05/14 15:05
     **/
    public String getAccessTokenOriginalContent(TokenInfoPojo tokenInfoPojo) {
        return tokenInfoPojo.getAccount() + "," + tokenInfoPojo.getAccountType() + ",accessToken," + System.currentTimeMillis();
    }

    /**
     * description 获取刷新Token原始内容
     * param [tokenInfoPojo]
     * return java.lang.String
     * author Administrator
     * createTime 2020/05/14 15:06
     **/
    public String getRefreshTokenOriginalContent(TokenInfoPojo tokenInfoPojo) {
        return tokenInfoPojo.getAccount() + "," + tokenInfoPojo.getAccountType() + ",refreshToken," + System.currentTimeMillis();
    }

    /**
     * description 获取Token信息缓存Key
     * param [account, accountType]
     * return java.lang.String
     * author Administrator
     * createTime 2020/05/14 15:07
     **/
    public String getTokenInfoCacheKey(String account, Integer accountType) {
        return applicationName + "::token::" + accountType + "::" + account;
    }

    /**
     * description 清除缓存Key
     * param [account, accountType]
     * return java.lang.String
     * author Administrator
     * createTime 2020/05/14 15:07
     **/
    public void cleatTokenInfoCache(TokenInfoPojo tokenInfoPojo) {
        String key =getTokenInfoCacheKey(tokenInfoPojo.getAccount(), tokenInfoPojo.getAccountType());
        redisComponent.del(key);
    }
}
