package com.esv.freight.app.module.complaint.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.FeignUtils;
import com.esv.freight.app.common.validator.groups.ValidatorDetail;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.common.validator.groups.ValidatorList;
import com.esv.freight.app.common.validator.groups.ValidatorUpdate;
import com.esv.freight.app.feign.OwnerComplaintInterface;
import com.esv.freight.app.module.complaint.form.ComplaintForm;
import com.esv.freight.app.module.complaint.pojo.OwnerComplaintAdvicePojo;
import com.esv.freight.app.module.complaint.vo.ComplaintDetailVO;
import com.esv.freight.app.module.complaint.vo.ComplaintListItemVO;
import com.esv.freight.app.module.complaint.vo.ComplaintListVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 货主投诉Controller
 * @project: freight-app-service
 * @name: com.esv.freight.app.module.complaint.controller.OwnerComplaintController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/05/28 13:30
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/complaint")
@Slf4j
@Validated
public class OwnerComplaintController {

    private OwnerComplaintInterface ownerComplaintInterface;

    @Autowired
    public OwnerComplaintController(OwnerComplaintInterface ownerComplaintInterface) {
        this.ownerComplaintInterface = ownerComplaintInterface;
    }

    /**
     * description 获取投诉列表
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/28 14:30
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody(required=false) @Validated(ValidatorList.class) ComplaintForm complaintForm) {

        // 调用投诉列表分页查询接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("goodsOwnerName", complaintForm.getGoodsOwnerName());
        reqJson.put("pageNum", complaintForm.getPageNum());
        reqJson.put("pageSize", complaintForm.getPageSize());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerComplaintInterface.list(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        ComplaintListVO pageResultVO = new ComplaintListVO();
        JSONObject data = result.getJSONObject("data");
        pageResultVO.setPageSize(data.getLong("pageSize"));
        pageResultVO.setPageNum(data.getLong("pageNum"));
        pageResultVO.setTotal(data.getLong("totalSize"));
        pageResultVO.setRecordSize(data.getLong("recordSize"));
        if(pageResultVO.getRecordSize() > 0) {
            JSONArray items = data.getJSONArray("records");
            List<ComplaintListItemVO> records = new ArrayList<>();
            for(int i=0; i<items.size(); ++i) {
                ComplaintListItemVO vo = new ComplaintListItemVO();
                JSONObject object = items.getJSONObject(i);
                vo.setId(object.getLong("id"));
                vo.setComplaintNo(object.getString("complaintNo"));
                vo.setWaybillId(object.getLong("waybillId"));
                vo.setWaybillNo(object.getString("waybillNo"));
                vo.setComplaintState(object.getInteger("state"));
                records.add(vo);
            }

            pageResultVO.setRecord(records);
        }
        return EResponse.ok(pageResultVO);
    }

    /**
     * description 获取投诉详情
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/28 14:30
     **/
    @RequestMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) ComplaintForm complaintForm) {

        // 调用投诉详情接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", complaintForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerComplaintInterface.detail(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        OwnerComplaintAdvicePojo ownerComplaintAdvicePojo = JSONObject.toJavaObject(FeignUtils.getFeignDataJson(result), OwnerComplaintAdvicePojo.class);
        ComplaintDetailVO complaintDetailVO = new ComplaintDetailVO();
        BeanUtils.copyProperties(ownerComplaintAdvicePojo, complaintDetailVO);
        complaintDetailVO.setComplaintState(ownerComplaintAdvicePojo.getState());
        return EResponse.ok(complaintDetailVO);
    }

    /**
     * description 添加投诉
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/28 15:00
     **/
    @RequestMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) ComplaintForm complaintForm) {

        // 调用添加投诉接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("complaintAdvice", 1);
        reqJson.put("state", 1);
        reqJson.put("goodsOwnerId", complaintForm.getUserId());
        reqJson.put("goodsOwnerName", complaintForm.getGoodsOwnerName());
        reqJson.put("waybillId", complaintForm.getWaybillId());
        reqJson.put("complaintObject", complaintForm.getComplaintObject());
        reqJson.put("complaintType", complaintForm.getComplaintType());
        reqJson.put("complaintContent", complaintForm.getComplaintContent());
        if(!StringUtils.isEmpty(complaintForm.getPictureUrl01())) {
            reqJson.put("pictureUrl01", complaintForm.getPictureUrl01());
        }
        if(!StringUtils.isEmpty(complaintForm.getPictureUrl02())) {
            reqJson.put("pictureUrl02", complaintForm.getPictureUrl02());
        }
        log.info(reqJson.toJSONString());
        JSONObject result = ownerComplaintInterface.add(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        return EResponse.ok();
    }

    /**
     * description 修改投诉
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/28 15:00
     **/
    @RequestMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) ComplaintForm complaintForm) {

        // 调用编辑投诉接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", complaintForm.getId());
        reqJson.put("goodsOwnerId", complaintForm.getUserId());
        reqJson.put("goodsOwnerName", complaintForm.getGoodsOwnerName());
        reqJson.put("complaintType", complaintForm.getComplaintType());
        reqJson.put("complaintContent", complaintForm.getComplaintContent());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerComplaintInterface.edit(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        return EResponse.ok();
    }

    /**
     * description 取消投诉
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/28 15:00
     **/
    @RequestMapping("/cancel")
    public EResponse cancel(@RequestBody @Validated(ValidatorDetail.class) ComplaintForm complaintForm) {

        // 调用取消投诉接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", complaintForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerComplaintInterface.cancel(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        return EResponse.ok();
    }
}
