package com.esv.freight.app.module.order.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.common.validator.groups.ValidatorDetail;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.common.validator.groups.ValidatorList;
import com.esv.freight.app.common.validator.groups.ValidatorUpdate;
import com.esv.freight.app.feign.TmsOrderInterface;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.order.form.OrderForm;
import com.esv.freight.app.module.order.form.OrderQueryForm;
import com.esv.freight.app.module.order.vo.OrderDetailVO;
import com.esv.freight.app.module.order.vo.OrderListItemVO;
import com.esv.freight.app.module.order.vo.OrderListVO;
import com.esv.freight.app.module.order.vo.OrderStatisticsVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 货主订单Controller
 * @project: freight-app-service
 * @name: com.esv.freight.app.module.order.controller.OrderController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/13 13:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/order")
@Slf4j
@Validated
public class OrderController {

    private TmsOrderInterface tmsInterface;

    @Autowired
    public OrderController(TmsOrderInterface tmsInterface) {
        this.tmsInterface = tmsInterface;
    }

    /**
     * description 获取订单分类统计信息
     * param param [accessToken]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/11 11:00
     **/
    @PostMapping("/statistics/getCountByType")
    public EResponse getCountByType() {

        // 调用货主查询不同状态的订单数接口
        JSONObject result = tmsInterface.getOrderCount();
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OrderStatisticsVO orderStatisticsVO = new OrderStatisticsVO();
        JSONObject data = result.getJSONObject("data");
        orderStatisticsVO.setCancelled(data.getInteger("cancelledCount"));
        orderStatisticsVO.setComplete(data.getInteger("finishedCount"));
        orderStatisticsVO.setExecuting(data.getInteger("executingCount"));
        orderStatisticsVO.setPending(data.getInteger("auditPendingCount"));
        orderStatisticsVO.setPublished(data.getInteger("addedCount"));
        return EResponse.ok(orderStatisticsVO);
    }

    /**
     * description 获取订单列表
     * param [accessToken, orderQueryForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/18 11:00
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody(required=false) @Validated(ValidatorList.class) OrderQueryForm orderQueryForm) {

        // 调用订单列表分页查询接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("goodsOwnerId", orderQueryForm.getGoodsOwnerId());
        reqJson.put("pageNum", orderQueryForm.getPageNum());
        reqJson.put("pageSize", orderQueryForm.getPageSize());
        reqJson.put("orderStateList", orderQueryForm.getOrderStateList());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.listOrders(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OrderListVO pageResultVO = new OrderListVO();
        JSONObject data = result.getJSONObject("data");
        pageResultVO.setPageSize(data.getLong("pageSize"));
        pageResultVO.setPageNum(data.getLong("pageNum"));
        pageResultVO.setTotal(data.getLong("totalSize"));
        pageResultVO.setRecordSize(data.getLong("recordSize"));
        if(pageResultVO.getRecordSize() > 0) {
            JSONArray items = data.getJSONArray("records");
            List<OrderListItemVO> records = new ArrayList<>();
            for(int i=0; i<items.size(); ++i) {
                OrderListItemVO vo = new OrderListItemVO();
                JSONObject object = items.getJSONObject(i);
                vo.setId(object.getLong("id"));
                vo.setOrderNo(object.getString("orderNo"));
                vo.setOrderState(object.getInteger("orderState"));
                vo.setAuditState(object.getInteger("auditState"));
                vo.setDeliveryCityCode(object.getString("deliveryCityCode"));
                vo.setReceiveCityCode(object.getString("receiveCityCode"));
                vo.setBusinessType(object.getInteger("businessType"));
                vo.setGoodsType(object.getInteger("goodsType"));
                vo.setGoodsName(object.getInteger("goodsName"));
                vo.setGoodsNameCode(object.getInteger("goodsNameCode"));
                vo.setGoodsDetail(object.getString("goodsDetail"));
                vo.setTotalGoodsAmount(object.getBigDecimal("totalGoodsAmount"));
                vo.setRemainingGoodsAmount(object.getBigDecimal("remainingGoodsAmount"));
                vo.setGoodsUnit(Integer.parseInt(object.getString("goodsUnit")));
                vo.setGoodsUnitPrice(object.getBigDecimal("goodsUnitPrice"));
                vo.setFreightUnitPriceInput(object.getBigDecimal("freightUnitPriceInput"));
                vo.setRequiredDeliveryTime(object.getLong("requiredDeliveryTime"));
                vo.setRequiredReceiveTime(object.getLong("requiredReceiveTime"));
                vo.setCreateTime(object.getLong("createTime"));
                vo.setOrderSource(object.getString("orderSource"));
                vo.setPlaceOrderTime(object.getLong("placeOrderTime"));
                records.add(vo);
            }

            pageResultVO.setRecord(records);
        }
        return EResponse.ok(pageResultVO);
    }

    /**
     * description 发布订单
     * param [accessToken, orderForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 14:00
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody(required=false) @Validated(ValidatorInsert.class) OrderForm orderForm) {

        // 调用订单新增接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("goodsOwnerId", orderForm.getGoodsOwnerId());
        reqJson.put("businessType", orderForm.getBusinessType());
        reqJson.put("goodsType", orderForm.getGoodsType());
        reqJson.put("goodsNameCode", orderForm.getGoodsNameCode());
        reqJson.put("goodsDetail", orderForm.getGoodsDetail());
        reqJson.put("totalGoodsAmount", orderForm.getGoodsTotalAmount());
        reqJson.put("goodsUnit", orderForm.getGoodsUnit());
        reqJson.put("goodsUnitPrice", orderForm.getGoodsUnitPrice());
        reqJson.put("freightUnitPriceInput", orderForm.getFreightUnitPriceInput());
        reqJson.put("deliveryAddressId", orderForm.getDeliveryAddressId());
        reqJson.put("receiveAddressId", orderForm.getReceiveAddressId());
        reqJson.put("requiredDeliveryTime", orderForm.getRequiredDeliveryTime());
        reqJson.put("requiredReceiveTime", orderForm.getRequiredReceiveTime());
        reqJson.put("vehicleTypeMain", orderForm.getVehicleTypeMain());
        reqJson.put("vehicleTypeSub", orderForm.getVehicleTypeSub());
        reqJson.put("vehicleLength", orderForm.getVehicleLength());
        reqJson.put("orderSource", 2);
        reqJson.put("remark", orderForm.getRemark());
   //     reqJson.put("signData", orderForm.getSignData());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.addOrderOwner(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 取消订单
     * param [accessToken, orderQueryForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/21 14:00
     **/
    @PostMapping("/cancel")
    public EResponse cancel(@RequestBody(required=false) @Validated(ValidatorDetail.class) OrderQueryForm orderQueryForm) {

        // 调用取消订单接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("orderId", orderQueryForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.cancelOrder(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 获取订单详情
     * param [accessToken, orderQueryForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/21 14:00
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody(required=false) @Validated(ValidatorDetail.class) OrderQueryForm orderQueryForm) {

        // 调用获取订单详情接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("orderId", orderQueryForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.getOrderDetail(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OrderDetailVO orderDetailVO = new OrderDetailVO();
        JSONObject data = result.getJSONObject("data");
        orderDetailVO.setId(data.getLong("id"));
        orderDetailVO.setOrderNo(data.getString("orderNo"));
        orderDetailVO.setOrderState(data.getInteger("orderState"));
        orderDetailVO.setAuditState(data.getInteger("auditState"));
        orderDetailVO.setBusinessType(data.getInteger("businessType"));
        orderDetailVO.setGoodsType(data.getInteger("goodsType"));
        orderDetailVO.setGoodsNameCode(data.getInteger("goodsNameCode"));
        orderDetailVO.setGoodsName(data.getString("goodsName"));
        orderDetailVO.setGoodsDetail(data.getString("goodsDetail"));
        orderDetailVO.setGoodsTotalAmount(data.getBigDecimal("totalGoodsAmount"));
        orderDetailVO.setRemainingGoodsAmount(data.getBigDecimal("remainingGoodsAmount"));
        orderDetailVO.setGoodsUnit(data.getInteger("goodsUnit"));
        orderDetailVO.setGoodsUnitPrice(data.getBigDecimal("goodsUnitPrice"));
        orderDetailVO.setFreightUnitPriceInput(data.getBigDecimal("freightUnitPriceInput"));
        orderDetailVO.setDeliveryAddressId(data.getLong("deliveryAddressId"));
        orderDetailVO.setDeliveryAddressName(data.getString("deliveryAddressName"));
        orderDetailVO.setDeliveryProvinceCode(data.getString("deliveryProvinceCode"));
        orderDetailVO.setDeliveryCityCode(data.getString("deliveryCityCode"));
        orderDetailVO.setDeliveryDistrictCode(data.getString("deliveryDistrictCode"));
        orderDetailVO.setDeliveryDetailAddress(data.getString("deliveryDetailAddress"));
        orderDetailVO.setDeliverer(data.getString("deliverer"));
        orderDetailVO.setDelivererPhone(data.getString("delivererPhone"));
        orderDetailVO.setReceiveAddressId(data.getLong("receiveAddressId"));
        orderDetailVO.setReceiveAddressName(data.getString("receiveAddressName"));
        orderDetailVO.setReceiveProvinceCode(data.getString("receiveProvinceCode"));
        orderDetailVO.setReceiveCityCode(data.getString("receiveCityCode"));
        orderDetailVO.setReceiveDistrictCode(data.getString("receiveDistrictCode"));
        orderDetailVO.setReceiveDetailAddress(data.getString("receiveDetailAddress"));
        orderDetailVO.setReceiver(data.getString("receiver"));
        orderDetailVO.setReceiverPhone(data.getString("receiverPhone"));
        orderDetailVO.setRequiredDeliveryTime(data.getLong("requiredDeliveryTime"));
        orderDetailVO.setRequiredReceiveTime(data.getLong("requiredReceiveTime"));
        orderDetailVO.setVehicleTypeMain(data.getInteger("vehicleTypeMain"));
        orderDetailVO.setVehicleTypeSub(data.getInteger("vehicleTypeSub"));
        orderDetailVO.setVehicleLength(data.getBigDecimal("vehicleLength"));
        orderDetailVO.setRemark(data.getString("remark"));
        orderDetailVO.setPlaceOrderTime(data.getLong("placeOrderTime"));
        orderDetailVO.setRequiredReceiptTime(data.getLong("requiredReceiptTime"));
        return EResponse.ok(orderDetailVO);
    }

    /**
     * description 编辑订单
     * param [accessToken, orderForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/21 14:00
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody(required=false) @Validated(ValidatorUpdate.class) OrderForm orderForm) {

        // 调用编辑订单接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", orderForm.getId());
        reqJson.put("goodsOwnerId", orderForm.getGoodsOwnerId());
        reqJson.put("businessType", orderForm.getBusinessType());
        reqJson.put("goodsType", orderForm.getGoodsType());
        reqJson.put("goodsNameCode", orderForm.getGoodsNameCode());
        reqJson.put("goodsDetail", orderForm.getGoodsDetail());
        reqJson.put("totalGoodsAmount", orderForm.getGoodsTotalAmount());
        reqJson.put("goodsUnit", orderForm.getGoodsUnit());
        reqJson.put("goodsUnitPrice", orderForm.getGoodsUnitPrice());
        reqJson.put("freightUnitPriceInput", orderForm.getFreightUnitPriceInput());
        reqJson.put("deliveryAddressId", orderForm.getDeliveryAddressId());
        reqJson.put("receiveAddressId", orderForm.getReceiveAddressId());
        reqJson.put("requiredDeliveryTime", orderForm.getRequiredDeliveryTime());
        reqJson.put("requiredReceiveTime", orderForm.getRequiredReceiveTime());
        reqJson.put("vehicleTypeMain", orderForm.getVehicleTypeMain());
        reqJson.put("vehicleTypeSub", orderForm.getVehicleTypeSub());
        reqJson.put("vehicleLength", orderForm.getVehicleLength());
        reqJson.put("remark", orderForm.getRemark());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.editOrderOwner(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }
}
