package com.esv.freight.app.module.bill.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.util.FeignUtils;
import com.esv.freight.app.common.validator.groups.ValidatorDetail;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.common.validator.groups.ValidatorList;
import com.esv.freight.app.feign.OwnerBillInterface;
import com.esv.freight.app.feign.PayInterface;
import com.esv.freight.app.module.account.pojo.DriverAccountInfoPojo;
import com.esv.freight.app.module.bill.form.BillQueryForm;
import com.esv.freight.app.module.bill.form.OwnerPayForm;
import com.esv.freight.app.module.bill.pojo.OwnerBillDetailPojo;
import com.esv.freight.app.module.bill.vo.BillDetailVO;
import com.esv.freight.app.module.bill.vo.BillListItemVO;
import com.esv.freight.app.module.bill.vo.BillListVO;
import com.esv.freight.app.module.bill.vo.PayVO;
import com.esv.freight.app.module.evaluate.form.OwnerEvaluateForm;
import com.esv.freight.app.module.evaluate.vo.EvaluateListItemVO;
import com.esv.freight.app.module.evaluate.vo.EvaluateListVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 货主账单Controller
 * @project: freight-app-service
 * @name: com.esv.freight.app.module.bill.controller.OwnerBillController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/06/02 17:40
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/bill")
@Slf4j
@Validated
public class OwnerBillController {

    private OwnerBillInterface ownerBillInterface;
    private PayInterface payInterface;

    @Autowired
    public OwnerBillController(OwnerBillInterface ownerBillInterface, PayInterface payInterface) {
        this.ownerBillInterface = ownerBillInterface;
        this.payInterface = payInterface;
    }

    /**
     * description 创建支付订单
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/06/03 17:00
     **/
    @PostMapping("/createPayOrder")
    public EResponse createPayOrder(@RequestBody(required=false) @Validated(ValidatorInsert.class) OwnerPayForm ownerPayForm) {

        // 调用确认账单接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("customerId", ownerPayForm.getCustomerId());
        reqJson.put("billId", ownerPayForm.getBillNo());
        reqJson.put("billDesc", "应付账单");
        reqJson.put("txnAmt", ownerPayForm.getTxnAmt());
        log.info(reqJson.toJSONString());
        JSONObject result = payInterface.createPayOrder(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        PayVO payVO = new PayVO();
        payVO.setTn(result.getJSONObject("data").getString("tn"));
        return EResponse.ok(payVO);
    }

    /**
     * description 确认对账
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/06/03 13:30
     **/
    @PostMapping("/confirmBill")
    public EResponse confirmBill(@RequestBody(required=false) @Validated(ValidatorDetail.class) BillQueryForm billQueryForm) {

        // 调用确认账单接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", billQueryForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerBillInterface.confirmBill(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        return EResponse.ok();
    }

    /**
     * description 获取对账详情
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/06/03 13:30
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody(required=false) @Validated(ValidatorDetail.class) BillQueryForm billQueryForm) {

        // 调用账单详情接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", billQueryForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerBillInterface.billOwnerDetail(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        BillDetailVO billDetailVO = new BillDetailVO();
        OwnerBillDetailPojo ownerBillDetailPojo = JSONObject.toJavaObject(FeignUtils.getFeignDataJson(result), OwnerBillDetailPojo.class);
        BeanUtils.copyProperties(ownerBillDetailPojo, billDetailVO);
        billDetailVO.setBillNo(ownerBillDetailPojo.getRecvBillNo());
        billDetailVO.setBillState(ownerBillDetailPojo.getRecvBillState());
        return EResponse.ok(billDetailVO);
    }

    /**
     * description 获取对账列表
     * param complaintForm
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/06/02 18:00
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody(required=false) @Validated(ValidatorList.class) BillQueryForm billQueryForm) {

        // 调用评价列表分页查询接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("goodsOwnerId", billQueryForm.getUserId());
        reqJson.put("pageNum", billQueryForm.getPageNum());
        reqJson.put("pageSize", billQueryForm.getPageSize());
        log.info(reqJson.toJSONString());
        JSONObject result = ownerBillInterface.listBillOwner(reqJson);
        log.info(result.toJSONString());

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        BillListVO pageResultVO = new BillListVO();
        JSONObject data = result.getJSONObject("data");
        pageResultVO.setPageSize(data.getLong("pageSize"));
        pageResultVO.setPageNum(data.getLong("pageNum"));
        pageResultVO.setTotal(data.getLong("total"));
        pageResultVO.setRecordSize(data.getLong("recordSize"));
        if(pageResultVO.getRecordSize() > 0) {
            JSONArray items = data.getJSONArray("records");
            List<BillListItemVO> records = new ArrayList<>();
            for(int i=0; i<items.size(); ++i) {
                BillListItemVO vo = new BillListItemVO();
                JSONObject object = items.getJSONObject(i);
                vo.setId(object.getLong("id"));
                vo.setBillNo(object.getString("recvBillNo"));
                vo.setBillState(object.getInteger("recvBillState"));
                vo.setBillTimeBucketStart(object.getDate("billTimeBucketStart"));
                vo.setBillTimeBucketEnd(object.getDate("billTimeBucketEnd"));
                vo.setTotalMoney(object.getBigDecimal("totalMoney"));
                records.add(vo);
            }

            pageResultVO.setRecord(records);
        }
        return EResponse.ok(pageResultVO);
    }
}
