package com.esv.freight.app.module.account.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.feign.DictInterface;
import com.esv.freight.app.feign.FileInterface;
import com.esv.freight.app.feign.GoodsOwnerInterface;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.form.*;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginByPwd;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginBySms;
import com.esv.freight.app.module.account.vo.OwnerAccountDetailVO;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.module.account.vo.LoginVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * @description: 账号Controller
 * @project: Freight
 * @name: com.esv.freight.module.ownerBackend.account.controller.OwnerAccountController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/09 17:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/account")
@Slf4j
@Validated
public class OwnerAccountController {

    private DictInterface dictInterface;
    private NoticeInterface noticeInterface;
    private GoodsOwnerInterface goodsOwnerInterface;
    private AppLoginService appLoginService;
    private FileInterface fileInterface;

    @Autowired
    public OwnerAccountController(FileInterface fileInterface, GoodsOwnerInterface goodsOwnerInterface, DictInterface dictInterface, NoticeInterface noticeInterface, AppLoginService appLoginService) {
        this.dictInterface = dictInterface;
        this.goodsOwnerInterface = goodsOwnerInterface;
        this.noticeInterface = noticeInterface;
        this.appLoginService = appLoginService;
        this.fileInterface = fileInterface;
    }

    /**
     * description 短信验证码登录
     * param [LoginBySmsForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/10 16:00
     **/
    @PostMapping("/login/loginBySms")
    public EResponse loginBySms(@RequestBody(required=false) @Validated(ValidatorLoginBySms.class) LoginForm loginForm) {

        // 调用通知服务验证短信接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", loginForm.getPhone());
        reqJson.put("type", "login");
        reqJson.put("captcha", loginForm.getSmsCode());
        JSONObject resultCheck = noticeInterface.checkSmsCaptcha(reqJson);

        if(resultCheck.getInteger("code") != 200) {
            return EResponse.error(resultCheck.getInteger("code"), resultCheck.getString("message"));
        }

        // 调用注册帐号接口
        reqJson.clear();
        reqJson.put("account", loginForm.getPhone());
        JSONObject resultRegister = goodsOwnerInterface.accountRegister(reqJson);

        // 1001表示 帐号已存在
        if(resultRegister.getInteger("code") != 200 && resultRegister.getInteger("code") != 1001) {
            return EResponse.error(resultRegister.getInteger("code"), resultRegister.getString("message"));
        }

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号密码登录RequestHeader
     * param [LoginByPwdForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/10 16:00
     **/
    @PostMapping("/login/loginByPwd")
    public EResponse loginByPwd(@RequestBody(required=false) @Validated(ValidatorLoginByPwd.class) LoginForm loginForm) {

        // 调用帐号密码校验接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("account", loginForm.getPhone());
        reqJson.put("password", loginForm.getPwd());
        JSONObject result = goodsOwnerInterface.checkAccountPwd(reqJson);

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        LoginVO loginByPwdVO = appLoginService.login(loginForm.getPhone());
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 帐号登出
     * param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/11 11:00
     **/
    @PostMapping("/logout")
    public EResponse logout(@RequestHeader("Union-Authorization") String accessToken) {
        appLoginService.logout(accessToken);
        return EResponse.ok();
    }

    /**
     * description 刷新Token
     * param [refreshToken] 刷新Token
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 09:00
     **/
    @PostMapping("/token/refresh")
    public EResponse refresh(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) RefreshTokenForm refreshTokenForm) {

        LoginVO loginByPwdVO = appLoginService.refreshToken(accessToken, refreshTokenForm);
        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 获取帐号详情
     * param []
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestHeader("Union-Authorization") String accessToken) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        String phone = appLoginService.getPhoneByAccessToken(accessToken);

        // 调用帐号密码校验接口
        JSONObject reqJsonDetail = new JSONObject();
        reqJsonDetail.put("account", phone);
        log.info(reqJsonDetail.toJSONString());
        JSONObject result = goodsOwnerInterface.getAccountDetail(reqJsonDetail);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OwnerAccountDetailVO accountDetailVO = new OwnerAccountDetailVO();
        accountDetailVO.setId(String.valueOf(result.getJSONObject("data").getLong("id")));
        accountDetailVO.setAccount(result.getJSONObject("data").getString("account"));
        accountDetailVO.setSourceType(result.getJSONObject("data").getString("sourceType"));
        accountDetailVO.setAuditStatus(result.getJSONObject("data").getInteger("auditStatus"));
        accountDetailVO.setOwnerType(result.getJSONObject("data").getInteger("ownerType"));
        accountDetailVO.setName(result.getJSONObject("data").getString("contactor"));
        accountDetailVO.setIdCard(result.getJSONObject("data").getString("idCard"));
        accountDetailVO.setIdCardExpireDate(result.getJSONObject("data").getString("idCardExpireDate"));
        accountDetailVO.setIdCardFileFrontId(result.getJSONObject("data").getString("idCardFrontUrl"));
        accountDetailVO.setIdCardFileBackId(result.getJSONObject("data").getString("idCardBackUrl"));
        accountDetailVO.setProvinceCode(result.getJSONObject("data").getString("provinceCode"));
        accountDetailVO.setCityCode(result.getJSONObject("data").getString("cityCode"));
        accountDetailVO.setDistrictCode(result.getJSONObject("data").getString("districtCode"));
        accountDetailVO.setDetailAddress(result.getJSONObject("data").getString("detailAddress"));
        return EResponse.ok(accountDetailVO);
    }

    /**
     * description 帐号实名认证
     * param [RealNameAuthForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/realNameAuth")
    public EResponse realNameAuth(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) RealNameAuthForm realNameAuthForm) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用上传单个文件接口-身份证正面
        JSONObject reqJson = new JSONObject();
        reqJson.put("fileType", "image");
        reqJson.put("fileData", realNameAuthForm.getIdCardFileFront());
        reqJson.put("fileName", "idCardFileFront");
        log.info(reqJson.toJSONString());
        JSONObject result = fileInterface.uploadSingleFile(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        String idCardFrontId = String.valueOf(result.getJSONObject("data").getString("id"));

        // 调用上传单个文件接口-身份证反面
        reqJson.clear();
        reqJson.put("fileType", "image");
        reqJson.put("fileData", realNameAuthForm.getIdCardFileBack());
        reqJson.put("fileName", "idCardFileBack");
        log.info(reqJson.toJSONString());
        result = fileInterface.uploadSingleFile(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        String idCardBackId = String.valueOf(result.getJSONObject("data").getString("id"));

        // 调用编辑帐号信息接口
        reqJson.clear();
        reqJson.put("id", realNameAuthForm.getId());
        reqJson.put("ownerType", realNameAuthForm.getOwnerType());
        reqJson.put("contactor", realNameAuthForm.getName());
        reqJson.put("idCard", realNameAuthForm.getIdCard());
        reqJson.put("idCardExpireDate", realNameAuthForm.getIdCardExpireDate());
        reqJson.put("idCardFrontUrl", idCardFrontId);
        reqJson.put("idCardBackUrl", idCardBackId);
        reqJson.put("detailAddress", realNameAuthForm.getDetailAddress());
        reqJson.put("provinceCode", realNameAuthForm.getProvinceCode());
        reqJson.put("cityCode", realNameAuthForm.getCityCode());
        reqJson.put("districtCode", realNameAuthForm.getDistrictCode());
        log.info(reqJson.toJSONString());
        result = goodsOwnerInterface.updateAccountInfo(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }
}