package com.esv.freight.app.module.account.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.component.TokenComponent;
import com.esv.freight.app.common.constants.AccountConstants;
import com.esv.freight.app.common.util.FeignUtils;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.common.validator.groups.ValidatorUpdate;
import com.esv.freight.app.feign.GoodsOwnerInterface;
import com.esv.freight.app.feign.NoticeInterface;
import com.esv.freight.app.module.account.form.*;
import com.esv.freight.app.module.account.pojo.OwnerAccountInfoPojo;
import com.esv.freight.app.module.account.pojo.TokenInfoPojo;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.validator.groups.ValidatorAccountExist;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginByPwd;
import com.esv.freight.app.module.account.validator.groups.ValidatorLoginBySms;
import com.esv.freight.app.module.account.vo.OwnerAccountDetailVO;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.module.account.vo.LoginVO;
import com.esv.freight.app.module.message.form.AccountDeviceForm;
import com.esv.freight.app.module.message.service.AccountDeviceInfoService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * @description: 账号Controller
 * @project: freight-app-service
 * @name: com.esv.freight.module.ownerBackend.account.controller.OwnerAccountController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/09 17:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/account")
@Slf4j
@Validated
public class OwnerAccountController {

    private GoodsOwnerInterface goodsOwnerInterface;
    private NoticeInterface noticeInterface;
    private AppLoginService appLoginService;
    private TokenComponent tokenComponent;
    private AccountDeviceInfoService accountDeviceInfoService;

    @Autowired
    public OwnerAccountController(TokenComponent tokenComponent, AccountDeviceInfoService accountDeviceInfoService, GoodsOwnerInterface goodsOwnerInterface, NoticeInterface noticeInterface, AppLoginService appLoginService) {
        this.goodsOwnerInterface = goodsOwnerInterface;
        this.noticeInterface = noticeInterface;
        this.appLoginService = appLoginService;
        this.tokenComponent = tokenComponent;
        this.accountDeviceInfoService = accountDeviceInfoService;
    }

    /**
     * description 短信验证码登录
     * param [LoginBySmsForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/10 16:00
     **/
    @PostMapping("/login/loginBySms")
    public EResponse loginBySms(@RequestBody(required=false) @Validated(ValidatorLoginBySms.class) LoginForm loginForm) {

        // 调用通知服务验证短信接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", loginForm.getPhone());
        reqJson.put("type", "login");
        reqJson.put("captcha", loginForm.getSmsCode());
        JSONObject result;
        try {
            result = noticeInterface.checkSmsCaptcha(reqJson);
        } catch (Exception e) {
            log.error("Feign请求时发生错误：" + e.getMessage(), e);
            return EResponse.error();
        }

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        // 2:查询货主详情(通过帐号)
        reqJson.clear();
        reqJson.put("account", loginForm.getPhone());
        try {
            result = goodsOwnerInterface.getAccountDetail(reqJson);
        } catch (Exception e) {
            log.error("Feign请求时发生错误：" + e.getMessage(), e);
            return EResponse.error();
        }

        // 3:表示账号未注册
        if(result.getInteger("code") == 1002) {
            // 调用注册帐号接口
            reqJson.clear();
            reqJson.put("account", loginForm.getPhone());
            try {
                result = goodsOwnerInterface.accountRegister(reqJson);
            } catch (Exception e) {
                log.error("Feign请求时发生错误：" + e.getMessage(), e);
                return EResponse.error();
            }

            if(!FeignUtils.isFeignSuccess(result)) {
                return FeignUtils.getFeignEResponse(result);
            }

            OwnerAccountInfoPojo pojo = new OwnerAccountInfoPojo();
            pojo.setId(result.getJSONObject("data").getLong("id"));
            pojo.setAccount(loginForm.getPhone());
            pojo.setLoginMode(AccountConstants.ACCOUNT_LOGIN_MODE_SMS);
            LoginVO vo = appLoginService.ownerLogin(pojo);

            // 提交推送标识
            AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
            accountDeviceForm.setAccount(loginForm.getPhone());
            accountDeviceForm.setAccountId(pojo.getId());
            accountDeviceForm.setDeviceSn(loginForm.getPushId());
            accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

            return EResponse.ok(vo);
        }

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        // 校验帐号状态：1-正常、2-停用
        OwnerAccountInfoPojo ownerAccountInfoPojo = JSONObject.toJavaObject(FeignUtils.getFeignDataJson(result), OwnerAccountInfoPojo.class);
        if (AccountConstants.AUDIT_STATUS_CHECK_SUCCESS.equals(ownerAccountInfoPojo.getAuditStatus())) {
            // 校验帐号状态：1-正常、2-停用
            if (AccountConstants.ACCOUNT_STATUS_BLOCK.equals(ownerAccountInfoPojo.getAccountStatus())) {
                return EResponse.error(1004, "帐号已停用");
            }
        }

        // 登录
        ownerAccountInfoPojo.setLoginMode(AccountConstants.ACCOUNT_LOGIN_MODE_SMS);
        LoginVO vo = appLoginService.ownerLogin(ownerAccountInfoPojo);

        // 提交推送标识
        AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
        accountDeviceForm.setAccount(loginForm.getPhone());
        accountDeviceForm.setAccountId(ownerAccountInfoPojo.getId());
        accountDeviceForm.setDeviceSn(loginForm.getPushId());
        accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

        return EResponse.ok(vo);
    }

    /**
     * description 帐号密码登录RequestHeader
     * param [LoginByPwdForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/10 16:00
     **/
    @PostMapping("/login/loginByPwd")
    public EResponse loginByPwd(@RequestBody(required=false) @Validated(ValidatorLoginByPwd.class) LoginForm loginForm) {

        // 1:调用帐号密码校验接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("account", loginForm.getPhone());
        reqJson.put("password", loginForm.getPwd());
        JSONObject result;
        try {
            result = goodsOwnerInterface.checkAccountPwd(reqJson);
        } catch (Exception e) {
            log.error("Feign请求时发生错误：" + e.getMessage(), e);
            return EResponse.error();
        }

        if(!FeignUtils.isFeignSuccess(result)) {
            return FeignUtils.getFeignEResponse(result);
        }

        // 校验帐号状态：1-正常、2-停用
        OwnerAccountInfoPojo ownerAccountInfoPojo = JSONObject.toJavaObject(FeignUtils.getFeignDataJson(result), OwnerAccountInfoPojo.class);
        if (AccountConstants.AUDIT_STATUS_CHECK_SUCCESS.equals(ownerAccountInfoPojo.getAuditStatus())) {
            // 校验帐号状态：1-正常、2-停用
            if (AccountConstants.ACCOUNT_STATUS_BLOCK.equals(ownerAccountInfoPojo.getAccountStatus())) {
                return EResponse.error(1003, "帐号已停用");
            }
        }

        // 登录
        ownerAccountInfoPojo.setLoginMode(AccountConstants.ACCOUNT_LOGIN_MODE_PWD);
        LoginVO vo = appLoginService.ownerLogin(ownerAccountInfoPojo);

        // 提交推送标识
        AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
        accountDeviceForm.setAccount(loginForm.getPhone());
        accountDeviceForm.setAccountId(ownerAccountInfoPojo.getId());
        accountDeviceForm.setDeviceSn(loginForm.getPushId());
        accountDeviceInfoService.saveAccountDevice(accountDeviceForm);
        return EResponse.ok(vo);
    }

    /**
     * description 帐号登出
     * param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/11 11:00
     **/
    @PostMapping("/logout")
    public EResponse logout() {
        accountDeviceInfoService.accountLogout();
        appLoginService.logout();

        return EResponse.ok();
    }

    /**
     * description 刷新Token
     * param [refreshToken] 刷新Token
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 09:00
     **/
    @PostMapping("/token/refresh")
    public EResponse refresh(@RequestBody(required=false) @Validated(ValidatorUpdate.class) RefreshTokenForm refreshTokenForm) {

        LoginVO loginByPwdVO = appLoginService.refreshToken(refreshTokenForm);

        TokenInfoPojo tokenInfoPojo = tokenComponent.getTokenInfo();
        // 提交推送标识
        AccountDeviceForm accountDeviceForm = new AccountDeviceForm();
        accountDeviceForm.setAccount(tokenInfoPojo.getAccount());
        accountDeviceForm.setAccountId(tokenInfoPojo.getAccountId());
        accountDeviceForm.setDeviceSn(refreshTokenForm.getPushId());
        accountDeviceInfoService.saveAccountDevice(accountDeviceForm);

        return EResponse.ok(loginByPwdVO);
    }

    /**
     * description 获取帐号详情
     * param []
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/detail")
    public EResponse detail() {

        TokenInfoPojo tokenInfoPojo = tokenComponent.getTokenInfo();
        String phone = tokenInfoPojo.getAccount();

        // 调用获取账号详情接口
        JSONObject reqJsonDetail = new JSONObject();
        reqJsonDetail.put("account", phone);
        log.info(reqJsonDetail.toJSONString());
        JSONObject result = goodsOwnerInterface.getAccountDetail(reqJsonDetail);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OwnerAccountDetailVO accountDetailVO = new OwnerAccountDetailVO();
        accountDetailVO.setId(String.valueOf(result.getJSONObject("data").getLong("id")));
        accountDetailVO.setAccount(result.getJSONObject("data").getString("account"));
        accountDetailVO.setSourceType(result.getJSONObject("data").getString("sourceType"));
        accountDetailVO.setAuditStatus(result.getJSONObject("data").getInteger("auditStatus"));
        accountDetailVO.setOwnerType(result.getJSONObject("data").getInteger("ownerType"));
        accountDetailVO.setName(result.getJSONObject("data").getString("contactor"));
        accountDetailVO.setIdCard(result.getJSONObject("data").getString("idCard"));
        accountDetailVO.setIdCardExpireDate(result.getJSONObject("data").getString("idCardExpireDate"));
        accountDetailVO.setIdCardFileFrontUrl(result.getJSONObject("data").getString("idCardFrontUrl"));
        accountDetailVO.setIdCardFileBackUrl(result.getJSONObject("data").getString("idCardBackUrl"));
        accountDetailVO.setProvinceCode(result.getJSONObject("data").getString("provinceCode"));
        accountDetailVO.setCityCode(result.getJSONObject("data").getString("cityCode"));
        accountDetailVO.setDistrictCode(result.getJSONObject("data").getString("districtCode"));
        accountDetailVO.setDetailAddress(result.getJSONObject("data").getString("detailAddress"));
        accountDetailVO.setOwnerFullName(result.getJSONObject("data").getString("ownerFullName"));
        accountDetailVO.setOwnerBriefName(result.getJSONObject("data").getString("ownerBriefName"));
        accountDetailVO.setUniCreditCode(result.getJSONObject("data").getString("uniCreditCode"));
        accountDetailVO.setCreditExpireDate(result.getJSONObject("data").getString("creditExpireDate"));
        accountDetailVO.setCreditOriginalUrl(result.getJSONObject("data").getString("creditOriginalUrl"));
        accountDetailVO.setCreditCopyUrl(result.getJSONObject("data").getString("creditCopyUrl"));
        accountDetailVO.setLegalPerson(result.getJSONObject("data").getString("legalPerson"));
        accountDetailVO.setLegalPhone(result.getJSONObject("data").getString("legalPhone"));
        return EResponse.ok(accountDetailVO);
    }

    /**
     * description 帐号实名认证
     * param [RealNameAuthForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 10:00
     **/
    @PostMapping("/realNameAuth")
    public EResponse realNameAuth(@RequestBody(required=false) @Validated(ValidatorUpdate.class) OwnerAuthForm ownerAuthForm) {

        // 调用编辑帐号信息接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", ownerAuthForm.getId());
        reqJson.put("ownerType", ownerAuthForm.getOwnerType());
        reqJson.put("contactor", ownerAuthForm.getName());
        reqJson.put("idCard", ownerAuthForm.getIdCard());
        reqJson.put("idCardExpireDate", ownerAuthForm.getIdCardExpireDate());
        reqJson.put("idCardFrontUrl", ownerAuthForm.getIdCardFileFrontUrl());
        reqJson.put("idCardBackUrl", ownerAuthForm.getIdCardFileBackUrl());
        reqJson.put("detailAddress", ownerAuthForm.getDetailAddress());
        reqJson.put("provinceCode", ownerAuthForm.getProvinceCode());
        reqJson.put("cityCode", ownerAuthForm.getCityCode());
        reqJson.put("districtCode", ownerAuthForm.getDistrictCode());

        if("2".equals(ownerAuthForm.getOwnerType())) {
            reqJson.put("ownerFullName", ownerAuthForm.getOwnerFullName());
            reqJson.put("ownerBriefName", ownerAuthForm.getOwnerBriefName());
            reqJson.put("uniCreditCode", ownerAuthForm.getUniCreditCode());
            reqJson.put("creditExpireDate", ownerAuthForm.getCreditExpireDate());
            reqJson.put("creditOriginalUrl", ownerAuthForm.getCreditOriginalUrl());
            reqJson.put("creditCopyUrl", ownerAuthForm.getCreditCopyUrl());
            reqJson.put("legalPerson", ownerAuthForm.getLegalPerson());
            reqJson.put("legalPhone", ownerAuthForm.getLegalPhone());
        }

        log.info(reqJson.toJSONString());
        JSONObject result = goodsOwnerInterface.updateAccountInfo(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 货主账号停用
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/05/06 11:00
     **/
    @PostMapping("/stop")
    public EResponse stopUsingAccount(@RequestBody(required=false) @Validated(ValidatorAccountExist.class) LoginForm loginForm) {

        appLoginService.stopUsing(loginForm.getPhone(), AccountConstants.ACCOUNT_TYPE_GOODS_OWNER);
        return EResponse.ok();
    }
}