package com.esv.freight.app.module.order.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esv.freight.app.common.response.ECode;
import com.esv.freight.app.common.response.EResponse;
import com.esv.freight.app.common.validator.groups.ValidatorDetail;
import com.esv.freight.app.common.validator.groups.ValidatorInsert;
import com.esv.freight.app.common.validator.groups.ValidatorUpdate;
import com.esv.freight.app.feign.TmsInterface;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.order.form.OrderForm;
import com.esv.freight.app.module.order.form.OrderQueryForm;
import com.esv.freight.app.module.order.vo.OrderDetailVO;
import com.esv.freight.app.module.order.vo.OrderListItemVO;
import com.esv.freight.app.module.order.vo.OrderListVO;
import com.esv.freight.app.module.order.vo.OrderStatisticsVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @description: 货主订单Controller
 * @project: Freight
 * @name: com.esv.freight.app.module.order.controller.OrderController
 * @author: 张志臣
 * @email: zhangzhichen@esvtek.com
 * @createTime: 2020/04/13 13:00
 * @version:1.0
 */
@RestController
@RequestMapping("/ownerBackend/order")
@Slf4j
@Validated
public class OrderController {

    private AppLoginService appLoginService;
    private TmsInterface tmsInterface;

    private OrderListVO pageResultVerify,pageResultNew,pageResultExecute;

    /**
     * description 测试数据，临时使用
     * param param
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/11 11:00
     **/
    private void createTestData() {
        pageResultVerify = new OrderListVO();
        List<OrderListItemVO> verifyList = new ArrayList<>();
        OrderListItemVO itemVO = new OrderListItemVO();
        itemVO.setId(10056L);
        itemVO.setOrderNO("2020042109300101");
        itemVO.setOrderState(1);
        itemVO.setAuditState(2);
        itemVO.setDeliveryCity("210100");
        itemVO.setReceiveCity("210100");
        itemVO.setGoodsName(1101);
        itemVO.setGoodsTotalAmount(new BigDecimal(26));
        itemVO.setGoodsUnit("吨");

        itemVO.setCreateTime(new Date());
        itemVO.setRequiredReceiveTime(new Date());
        itemVO.setRequiredDeliveryTime(new Date());
        itemVO.setOrderSource("1");
        verifyList.add(itemVO);

        pageResultVerify.setPageNum(1L);
        pageResultVerify.setPageSize(20L);
        pageResultVerify.setTotal(1L);
        pageResultVerify.setRecordSize(1L);
        pageResultVerify.setRecord(verifyList);
    }

    @Autowired
    public OrderController(TmsInterface tmsInterface, AppLoginService appLoginService) {
        this.appLoginService = appLoginService;
        this.tmsInterface = tmsInterface;
        createTestData();
    }

    /**
     * description 获取订单分类统计信息
     * param param [accessToken]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/11 11:00
     **/
    @PostMapping("/statistics/getCountByType")
    public EResponse getCountByType(@RequestHeader("Union-Authorization") String accessToken) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        OrderStatisticsVO orderStatisticsVO = new OrderStatisticsVO();
        orderStatisticsVO.setCancelled(6);
        orderStatisticsVO.setComplete(5);
        orderStatisticsVO.setExecuting(3);
        orderStatisticsVO.setPending(6);
        orderStatisticsVO.setPublished(10);
        return EResponse.ok(orderStatisticsVO);
    }

    /**
     * description 获取订单列表
     * param [accessToken, orderQueryForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/18 11:00
     **/
    @PostMapping("/list")
    public EResponse list(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) OrderQueryForm orderQueryForm) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用订单列表分页查询接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("goodsOwnerId", orderQueryForm.getGoodsOwnerId());
        reqJson.put("pageNum", orderQueryForm.getPageNum());
        reqJson.put("pageSize", orderQueryForm.getPageSize());
        if(orderQueryForm.getOrderState() != null){
            reqJson.put("orderState", orderQueryForm.getOrderState());
        }
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.listOrders(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }
/*
        OrderListVO pageResultVO = new OrderListVO();
        JSONObject data = result.getJSONObject("data");
        pageResultVO.setPageSize(data.getLong("pageSize"));
        pageResultVO.setPageNum(data.getLong("pageNum"));
        pageResultVO.setTotal(data.getLong("total"));
        pageResultVO.setRecordSize(data.getLong("recordSize"));
        if(pageResultVO.getRecordSize() > 0) {
            JSONArray items = data.getJSONArray("records");
            List<OrderListItemVO> records = new ArrayList<>();
            for(int i=0; i<items.size(); ++i) {
                OrderListItemVO vo = new OrderListItemVO();
                JSONObject object = items.getJSONObject(i);
                vo.setId(object.getLong("id"));
                vo.setOrderNO(object.getString("orderNo"));
                vo.setOrderState(object.getInteger("orderState"));
                vo.setDeliveryCity(object.getString("deliveryCity"));
                vo.setReceiveCity(object.getString("receiveCity"));
                vo.setGoodsName(object.getInteger("goodsName"));
                vo.setGoodsTotalAmount(object.getBigDecimal("goodsTotalAmount"));
                vo.setGoodsUnit(object.getString("goodsUnit"));
                vo.setRequiredDeliveryTime(object.getDate("requiredDeliveryTime"));
                vo.setRequiredReceiveTime(object.getDate("requiredReceiveTime"));
                vo.setCreateTime(object.getDate("createTime"));
                records.add(vo);
            }

            pageResultVO.setRecord(records);
        }
        return EResponse.ok(pageResultVO);

*/
        return EResponse.ok(pageResultVerify);
    }

    /**
     * description 发布订单
     * param [accessToken, orderForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/13 14:00
     **/
    @PostMapping("/add")
    public EResponse add(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorInsert.class) OrderForm orderForm) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用订单新增接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("goodsOwnerId", orderForm.getGoodsOwnerId());
        reqJson.put("businessType", orderForm.getBusinessType());
        reqJson.put("goodsType", orderForm.getGoodsType());
        reqJson.put("goodsNameCode", orderForm.getGoodsNameCode());
        reqJson.put("goodsDetail", orderForm.getGoodsDetail());
        reqJson.put("goodsTotalAmount", orderForm.getGoodsTotalAmount());
        reqJson.put("goodsUnit", orderForm.getGoodsUnit());
        reqJson.put("goodsUnitPrice", orderForm.getGoodsUnitPrice());
        reqJson.put("freightUnitPriceInput", orderForm.getFreightUnitPriceInput());
        reqJson.put("deliveryAddressId", orderForm.getDeliveryAddressId());
        reqJson.put("receiveAddressId", orderForm.getReceiveAddressId());
        reqJson.put("requiredDeliveryTime", orderForm.getRequiredDeliveryTime());
        reqJson.put("requiredReceiveTime", orderForm.getRequiredReceiveTime());
        reqJson.put("vehicleTypeMain", orderForm.getVehicleTypeMain());
        reqJson.put("vehicleTypeSub", orderForm.getVehicleTypeSub());
        reqJson.put("vehicleLength", orderForm.getVehicleLength());
        reqJson.put("orderSource", 2);
        reqJson.put("remark", orderForm.getRemark());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.addOrderOwner(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 取消订单
     * param [accessToken, orderQueryForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/21 14:00
     **/
    @PostMapping("/cancel")
    public EResponse cancel(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorDetail.class) OrderQueryForm orderQueryForm) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用取消订单接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", orderQueryForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.cancelOrder(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }

    /**
     * description 获取订单详情
     * param [accessToken, orderQueryForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/21 14:00
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorDetail.class) OrderQueryForm orderQueryForm) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用获取订单详情接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", orderQueryForm.getId());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.getOrderDetail(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        OrderDetailVO orderDetailVO = new OrderDetailVO();
        JSONObject data = result.getJSONObject("data");
        orderDetailVO.setId(data.getLong("id"));
        orderDetailVO.setOrderNo(data.getString("orderNo"));
        orderDetailVO.setOrderState(data.getInteger("orderState"));
        orderDetailVO.setAuditState(data.getInteger("auditState"));
        orderDetailVO.setBusinessType(data.getInteger("businessType"));
        orderDetailVO.setGoodsType(data.getInteger("goodsType"));
        orderDetailVO.setGoodsNameCode(data.getInteger("goodsNameCode"));
        orderDetailVO.setGoodsNameCode(data.getInteger("goodsNameCode"));
        orderDetailVO.setGoodsDetail(data.getString("goodsDetail"));
        orderDetailVO.setGoodsTotalAmount(data.getBigDecimal("goodsTotalAmount"));
        orderDetailVO.setGoodsUnit(data.getInteger("goodsUnit"));
        orderDetailVO.setGoodsUnitPrice(data.getBigDecimal("goodsUnitPrice"));
        orderDetailVO.setFreightUnitPriceInput(data.getBigDecimal("freightUnitPriceInput"));
        orderDetailVO.setDeliveryAddressId(data.getLong("deliveryAddressId"));
        orderDetailVO.setReceiveAddressId(data.getLong("receiveAddressId"));
        orderDetailVO.setRequiredDeliveryTime(data.getLong("requiredDeliveryTime"));
        orderDetailVO.setRequiredReceiveTime(data.getLong("requiredReceiveTime"));
        orderDetailVO.setVehicleTypeMain(data.getInteger("vehicleTypeMain"));
        orderDetailVO.setVehicleTypeSub(data.getInteger("vehicleTypeSub"));
        orderDetailVO.setVehicleLength(data.getBigDecimal("vehicleLength"));
        orderDetailVO.setRemark(data.getString("remark"));
        return EResponse.ok(orderDetailVO);
    }

    /**
     * description 编辑订单
     * param [accessToken, orderForm]
     * return com.esv.freight.common.response.EResponse
     * author 张志臣
     * createTime 2020/04/21 14:00
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestHeader("Union-Authorization") String accessToken, @RequestBody(required=false) @Validated(ValidatorUpdate.class) OrderForm orderForm) {

        if(appLoginService.isInvalidAccessToken(accessToken)) {
            return EResponse.error(ECode.TOKEN_EXPIRED);
        }

        // 调用编辑订单接口
        JSONObject reqJson = new JSONObject();
        reqJson.put("id", orderForm.getId());
        reqJson.put("goodsOwnerId", orderForm.getGoodsOwnerId());
        reqJson.put("businessType", orderForm.getBusinessType());
        reqJson.put("goodsType", orderForm.getGoodsType());
        reqJson.put("goodsNameCode", orderForm.getGoodsNameCode());
        reqJson.put("goodsDetail", orderForm.getGoodsDetail());
        reqJson.put("goodsTotalAmount", orderForm.getGoodsTotalAmount());
        reqJson.put("goodsUnit", orderForm.getGoodsUnit());
        reqJson.put("goodsUnitPrice", orderForm.getGoodsUnitPrice());
        reqJson.put("freightUnitPriceInput", orderForm.getFreightUnitPriceInput());
        reqJson.put("deliveryAddressId", orderForm.getDeliveryAddressId());
        reqJson.put("receiveAddressId", orderForm.getReceiveAddressId());
        reqJson.put("requiredDeliveryTime", orderForm.getRequiredDeliveryTime());
        reqJson.put("requiredReceiveTime", orderForm.getRequiredReceiveTime());
        reqJson.put("vehicleTypeMain", orderForm.getVehicleTypeMain());
        reqJson.put("vehicleTypeSub", orderForm.getVehicleTypeSub());
        reqJson.put("vehicleLength", orderForm.getVehicleLength());
        reqJson.put("remark", orderForm.getRemark());
        log.info(reqJson.toJSONString());
        JSONObject result = tmsInterface.editOrder(reqJson);
        log.info(result.toJSONString());

        if(result.getInteger("code") != 200) {
            return EResponse.error(result.getInteger("code"), result.getString("message"));
        }

        return EResponse.ok();
    }
}
