package com.esv.freight.app.module.account.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.app.common.component.TokenComponent;
import com.esv.freight.app.common.constants.AccountConstants;
import com.esv.freight.app.common.exception.EException;
import com.esv.freight.app.common.util.ReqUtils;
import com.esv.freight.app.module.account.dao.AppLoginDao;
import com.esv.freight.app.module.account.entity.AppLoginEntity;
import com.esv.freight.app.module.account.form.RefreshTokenForm;
import com.esv.freight.app.module.account.pojo.DriverAccountInfoPojo;
import com.esv.freight.app.module.account.pojo.TokenInfoPojo;
import com.esv.freight.app.module.account.service.AppLoginService;
import com.esv.freight.app.module.account.vo.LoginVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.UUID;

@Service("appLoginService")
public class AppLoginServiceImpl extends ServiceImpl<AppLoginDao, AppLoginEntity> implements AppLoginService {

    @Autowired
    private TokenComponent tokenComponent;

    private void createToken(AppLoginEntity entity) {
        if(entity == null) {
            return;
        }

        entity.setAccessToken(UUID.randomUUID().toString().replaceAll("-", ""));
        entity.setRefreshToken(UUID.randomUUID().toString().replaceAll("-", ""));
    }

    @Override
    public LoginVO driverLogin(DriverAccountInfoPojo driverAccountInfoPojo) {
        // 1:初始化Token
        TokenInfoPojo tokenInfoPojo = new TokenInfoPojo();
        BeanUtils.copyProperties(driverAccountInfoPojo, tokenInfoPojo);
        tokenInfoPojo.setAccountId(driverAccountInfoPojo.getId());
        this.tokenComponent.generateTokenInfo(tokenInfoPojo);

        // 2:保存或更新登录信息
        AppLoginEntity record = this.baseMapper.selectOne(new QueryWrapper<AppLoginEntity>().lambda()
                .eq(AppLoginEntity::getAccountType, tokenInfoPojo.getAccountType())
                .eq(AppLoginEntity::getPhone, tokenInfoPojo.getAccount()));
        AppLoginEntity appLoginEntity = new AppLoginEntity();
        BeanUtils.copyProperties(tokenInfoPojo, appLoginEntity);
        appLoginEntity.setLoginStatus(AccountConstants.ACCOUNT_STATUS_LOGIN);
        appLoginEntity.setPhone(tokenInfoPojo.getAccount());
        if (null == record) {
            this.baseMapper.insert(appLoginEntity);
        } else {
            appLoginEntity.setId(record.getId());
            this.baseMapper.updateById(appLoginEntity);
        }

        // 3:返回Token
        LoginVO loginVO = new LoginVO();
        BeanUtils.copyProperties(tokenInfoPojo, loginVO);
        return loginVO;
    }

    @Override
    public LoginVO login(String phone) {

        AppLoginEntity appLoginEntity = this.getAccountByPhone(phone);
        if(null == appLoginEntity) {
            return addLogin(phone);
        }

        createToken(appLoginEntity);
        appLoginEntity.setLoginStatus(AccountConstants.ACCOUNT_STATUS_LOGIN);
        appLoginEntity.setLoginTime(new Date());
        appLoginEntity.setRefreshTime(new Date());
        appLoginEntity.setAccessTokenValidTime(getFuture(1));
        appLoginEntity.setRefreshTokenValidTime(getFuture(30));
        this.baseMapper.updateById(appLoginEntity);

        //数据转换
        LoginVO loginByPwdVO = new LoginVO();
        loginByPwdVO.setAccessToken(appLoginEntity.getAccessToken());
        loginByPwdVO.setRefreshToken(appLoginEntity.getRefreshToken());
        return loginByPwdVO;
    }

    private LoginVO addLogin(String phone) {
        AppLoginEntity appLoginEntity = new AppLoginEntity();
        appLoginEntity.setPhone(phone);
        createToken(appLoginEntity);
        appLoginEntity.setLoginStatus(AccountConstants.ACCOUNT_STATUS_LOGIN);
        appLoginEntity.setLoginTime(new Date());
        appLoginEntity.setRefreshTime(new Date());
        appLoginEntity.setLogoutTime(new Date());
        appLoginEntity.setAccessTokenValidTime(getFuture(1));
        appLoginEntity.setRefreshTokenValidTime(getFuture(30));
        this.baseMapper.insert(appLoginEntity);

        //数据转换
        LoginVO loginByPwdVO = new LoginVO();
        loginByPwdVO.setAccessToken(appLoginEntity.getAccessToken());
        loginByPwdVO.setRefreshToken(appLoginEntity.getRefreshToken());
        return loginByPwdVO;
    }

    @Override
    public void logout(String accessToken) {
        AppLoginEntity entity = this.getAccountByAccessToken(accessToken);
        if(entity == null) {
            throw new EException(602, "Token已过期,请重新登录");
        }

        entity.setAccessToken("");
        entity.setRefreshToken("");
        entity.setLoginStatus(AccountConstants.ACCOUNT_STATUS_LOGOUT);
        entity.setLogoutTime(new Date());
        this.baseMapper.updateById(entity);
    }

    @Override
    public void stopUsing(String phone) {
        AppLoginEntity entity = this.getAccountByPhone(phone);
        if(null == entity) {
            return;
        }

        entity.setAccessToken("");
        entity.setRefreshToken("");
        entity.setLoginStatus(AccountConstants.ACCOUNT_STATUS_LOGOUT);
        entity.setLogoutTime(new Date());
        this.baseMapper.updateById(entity);
    }

    @Override
    public LoginVO refreshToken(String accessToken, RefreshTokenForm refreshTokenForm) {
        AppLoginEntity entity = this.getAccountByAccessToken(accessToken);
        if(entity == null) {
            throw new EException(602, "Token已过期,请重新登录");
        }

        if(isInvalidRefreshToken(refreshTokenForm.getRefreshToken())) {
            throw new EException(602, "Token已过期,请重新登录");
        }

        createToken(entity);
        entity.setRefreshTime(new Date());
        entity.setLoginStatus(AccountConstants.ACCOUNT_STATUS_LOGIN);
        entity.setAccessTokenValidTime(getFuture(1));
        entity.setRefreshTokenValidTime(getFuture(30));
        this.baseMapper.updateById(entity);

        //数据转换
        LoginVO loginByPwdVO = new LoginVO();
        loginByPwdVO.setAccessToken(entity.getAccessToken());
        loginByPwdVO.setRefreshToken(entity.getRefreshToken());
        return loginByPwdVO;
    }

    @Override
    public void checkAccessToken() {
        String accessToken = ReqUtils.getTokenInfo().getAccessToken();
        String phone = ReqUtils.getTokenInfo().getAccount();

        AppLoginEntity entity = this.getAccountByPhone(phone);
        // 手机号尚未登录
        if(entity == null) {
            throw new EException(601, "无效的Token");
        }

        if(StringUtils.isEmpty(entity.getAccessToken())) {
            throw new EException(601, "您的账号信息发生变化，需要您重新登录");
        }

        if(!entity.getAccessToken().equals(accessToken)) {
            throw new EException(601, "您的帐号正在其他设备登录，若非本人操作，您的密码已泄漏，请修改登录密码，以保证帐号安全。");
        }

        Date current = new Date();
        Date accessDate = entity.getAccessTokenValidTime();

        if(current.after(accessDate)) {
            throw new EException(602, "Token已过期");
        }
    }

    @Override
    public boolean isInvalidRefreshToken(String refreshToken) {
        AppLoginEntity entity = this.getAccountByRefreshToken(refreshToken);
        if(entity == null) {
            throw new EException(602, "Token已过期,请重新登录");
        }

        Date current = new Date();
        Date refreshDate = entity.getRefreshTokenValidTime();

        if(current.after(refreshDate)) {
            return true;
        }

        return false;
    }

    @Override
    public AppLoginEntity getAccountByPhone(String phone) {
        QueryWrapper<AppLoginEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("phone", phone);
        return this.baseMapper.selectOne(queryWrapper);
    }

    private AppLoginEntity getAccountByAccessToken(String accessToken) {
        QueryWrapper<AppLoginEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("access_token", accessToken);
        return this.baseMapper.selectOne(queryWrapper);
    }

    private AppLoginEntity getAccountByRefreshToken(String refreshToken) {
        QueryWrapper<AppLoginEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select().eq("refresh_token", refreshToken);
        return this.baseMapper.selectOne(queryWrapper);
    }

    /**
     * 返回未来天数
     * @param days
     * @return
     */
    public Date getFuture(int days) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) + days);
        return calendar.getTime();
    }
}
